/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: arrowheads.c
 *            DESCRIPTION: handle drawing arrow heads for ArrowGFX plug-in
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define ARROWHEADS_C

/* === System Headers === */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <math.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "arrows.h"
#include "arrowheads.h"
#include "gfxcommon.h"
#include "debug.h"


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();
extern void GFXPreviewUpdate();


/* === Public routine prototypes === */


/* === Private routine prototypes === */
static void PreviewUpdate();
static void PreviewButtonPress();
static void DrawSelectionOutline();


/* === Global Variables === */
extern GtkDrawingArea	*preview;
extern GtkWidget			*tail_length_text;

int	selected_index=0;


/* === Static Variables === */
static GtkDrawingArea	*darea;

static int		width, height, rows, columns;
static int		colwidth, rowwidth;
static int		mouse_x=0;
static int		mouse_y=0;


static GdkGC	*gcWhite = NULL;
static GdkGC	*gcBlack = NULL;
static GdkGC	*gcDotted = NULL;



/*========================================================================
 *	Name:			GFXDrawArrowHeads
 *	Prototype:	void GFXDrawArrowHeads(GtkWidget *frame)
 *					
 *	Description:
 *	Create and populate the preview window of arrow head types.
 *
 *	Input Arguments:
 *		GtkWidget	*frame		frame inside which arrow heads are displayed.
 *
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *		For each arrowhead
 *			scale to fit preview subwindow (a button)
 *			draw into GtkPreview 
 *			place preview in new subwindow (ie a button)
 *	Restrictions:
 *	Notes:
 * This is just a preliminary - re: proof of concept - implementation.  A
 * much better method will be to use saved files that use the arrow file
 * format specified in arrowheads.h.
 *========================================================================*/
void
GFXDrawArrowHeads(
	GtkWidget	*frame
)
{
	GtkWidget	*event_box;
	GtkWidget	*inframe;

	gtk_widget_realize(frame);

	/*
	 * Initialize graphics contexts.
	 */
	if (gcWhite == NULL)
	{
		gcWhite = gdk_gc_new(frame->window);
		gdk_gc_copy( gcWhite, GTK_WIDGET(frame)->style->white_gc );

		gcBlack = gdk_gc_new(frame->window);
		gdk_gc_copy( gcBlack, GTK_WIDGET(frame)->style->black_gc );

		gcDotted = gdk_gc_new(frame->window);
		gdk_gc_copy( gcDotted, GTK_WIDGET(frame)->style->black_gc );
	}

	gtk_widget_unrealize(frame);

	/*
	 * Calculate drawing area information.
	 */
	rows = (int)((ARROW_DEFS-1)/ARROW_MAX_COLUMNS) + 1;
	columns = ARROW_MAX_COLUMNS;
	colwidth = ARROW_PREVIEW_BUTTON_WSIZE + (2 * ARROW_BOX_WIDTH);
	rowwidth = ARROW_PREVIEW_BUTTON_HSIZE + (2 * ARROW_BOX_HEIGHT);
	width = colwidth * ARROW_MAX_COLUMNS + SIDE_MARGINS;
	height = rowwidth * rows + TOP_MARGINS;

	/*
	 * We need an event box to handle the mouse clicks for the drawing
	 * area.
	 */
	inframe = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(inframe), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (frame), inframe);
	gtk_widget_show(inframe);

	event_box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER (inframe), event_box);
	gtk_widget_show((GtkWidget *)event_box);

	/*
	 * Allocate the drawing area and put it in the event box.
	 */
	darea = (GtkDrawingArea *)gtk_drawing_area_new();
	gtk_drawing_area_size(darea, width, height);
	gtk_container_add (GTK_CONTAINER (event_box), (GtkWidget *)darea);
	gtk_widget_show((GtkWidget *)darea);
	gtk_widget_realize((GtkWidget *)darea);

	/*
	 * Attach signal handlers to the event box.
	 */
	gtk_signal_connect(GTK_OBJECT(GTK_WIDGET(darea)), "expose_event",
			(GtkSignalFunc)PreviewUpdate,
			NULL);
	gtk_signal_connect(GTK_OBJECT(GTK_WIDGET(event_box)), "button_press_event",
			(GtkSignalFunc)PreviewButtonPress,
			NULL);

	selected_index=0;

}


/*========================================================================
 *	Name:			PreviewUpdate
 *	Prototype:	void PreviewUpdate(GtkWidget *preview)
 *					
 *	Description:
 *	Draw available arrow heads in their window.
 *
 *	Input Arguments:
 *		GtkWidget	*preview		preview widget that gets the drawn arrowheads
 *
 *	Output Arguments:
 *	Return Values:
 *
 * Global variables
 * arrow_hdrs						headers for arrow shapes (arrowheads.h)
 *	arrow_vertices					vertices of arrow shapes (arrowheads.h)
 *
 *	Method:
 *		For each arrowhead
 *			If arrow is the right size (initial version - all heads same size!)
 *				calculate current offset into pixmap region.
 *				draw arrow head into pixmap
 *		copy pixmap into preview
 *	Restrictions:
 *	Notes:
 *========================================================================*/
static void
PreviewUpdate(
	GtkWidget		*widget,
	GdkEventExpose	*event
)
{
#ifdef DEBUG
	char			*fname="PreviewUpdate()";
#endif

	GdkPoint			*pts;
	int				i, j;
	int				xoffset, yoffset;
	int				rowcount;
	int				row, col;

	DBGEnter();

	/*
	 * Fill background with white.
	 */
	gdk_draw_rectangle(
			darea->widget.window, gcWhite, TRUE,
			0, 0, width, height);

	/*
	 * Calculate initial offsets - ie margins around white background.
	 */
	xoffset = SIDE_MARGINS/2;
	yoffset = TOP_MARGINS/2;

	/*
	 * For each arrowhead ...
	 */
	rowcount = 0;
	for (i=0; i<ARROW_DEFS; i++)
	{
		/*
		 * Allocate an array of GdkPoint points.
		 */
		pts = (GdkPoint *)g_new(gint, arrow_hdrs[i].points*sizeof(GdkPoint));

		/*
		 * Copy point locations into the GdkPoint array.
		 */
		for (j=0; j<arrow_hdrs[i].points; j++)
		{
			pts[j].x = (gint16)arrow_vertices[i][j].x + xoffset + ARROW_BOX_WIDTH;
			pts[j].y = (gint16)arrow_vertices[i][j].y + yoffset + ARROW_BOX_HEIGHT;
		}
	
		/*
		 * Draw the polygon for this arrowhead.
		 */
		gdk_draw_polygon(
				darea->widget.window, gcBlack, TRUE,
				pts, arrow_hdrs[i].points);

		/*
		 * Free memory associated with the array of points.
		 */
		g_free(pts);

		/*
		 * Recalculate offsets.
		 */
		if ( (i!=0) && ( ( (i+1) % ARROW_MAX_COLUMNS) == 0 ) )
		{
			xoffset = SIDE_MARGINS/2;
			rowcount++;
		}
		else
			xoffset += arrow_hdrs[i].grid_size + (2 * ARROW_BOX_WIDTH);

		yoffset =  rowcount * rowwidth + TOP_MARGINS/2;

	}

	/*
	 * Redraw selection outlines.
	 */
	if ( selected_index != 0 )
	{
		row = (selected_index-1) / ARROW_MAX_COLUMNS;
		col = (selected_index-1) % ARROW_MAX_COLUMNS;

		xoffset = col * colwidth + SIDE_MARGINS/2;
		yoffset = row * rowwidth + TOP_MARGINS/2;

		gdk_draw_rectangle(
			darea->widget.window, gcBlack, FALSE,
			xoffset, yoffset, colwidth, rowwidth);
	}


	/*
	 * Flush screen updates.
	 */
	gdk_flush();

}


/*========================================================================
 *	Name:			PreviewButtonPress
 *	Prototype:	void PreviewButtonPress(GtkWidget *widget, GdkEventButton *event)
 *					
 *	Description:
 *	Select the arrowhead underneath the pointer.
 *
 *	Input Arguments:
 *		GtkWidget		*w					unused
 *		GdkEventButton	*event			event information
 *
 *	Output Arguments:
 *	Return Values:
 * Global variables
 *	Method:
 *	Restrictions:
 *	Notes:
 *========================================================================*/
static void
PreviewButtonPress(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
#ifdef DEBUG
	char			*fname="PreviewButtonPress()";
#endif

	char	buf[32];

	DBGEnter();

	/*
	 * Calculate the index of the arrow head based on button click
	 * location.  We're looking for a row and column here.
	 */
	mouse_x = event->x;
	mouse_y = event->y;

	DrawSelectionOutline();

	/*
	 * For some reason, this next line crashes on Solaris, but not on Linux.
	 * It might be a gtk-1.0.4 problem - I'm upgrading Solaris to 1.0.6 now.
	gtk_signal_emit_by_name(GTK_OBJECT(preview), "expose_event");
	 */

	GFXPreviewUpdate();

	if ( selected_index != 0 )
	{
		sprintf(buf, "%d", arrow_hdrs[selected_index-1].tail.length);
		gtk_entry_set_text(GTK_ENTRY(tail_length_text), buf);
	}
}


/*========================================================================
 *	Name:			DrawSelectionOutline
 *	Prototype:	void DrawSelectionOutline()
 *					
 *	Description:
 *	Draw the outlines around the old selection and the new selection.
 *
 *	Input Arguments:
 *		GtkWidget		*w					unused
 *		GdkEventButton	*event			event information
 *
 *	Output Arguments:
 *	Return Values:
 * Global variables
 *	Method:
 *	Restrictions:
 *	Notes:
 *========================================================================*/
static void
DrawSelectionOutline()
{
#ifdef DEBUG
	char			*fname="DrawSelectionOutline()";
#endif

	int		i;
	int		xoffset=0, yoffset=0;
	int		selected_row=0, selected_column=0;
	int		new_index;
	int		row, col;

	DBGEnter();

	/*
	 * If we're not inside the selectable area, do nothing.
	 */
	if ( mouse_x < SIDE_MARGINS/2 ) return;
	if ( mouse_y < TOP_MARGINS/2 ) return;

	for (i=1; i<columns+1; i++)
	{
		xoffset = i*rowwidth + SIDE_MARGINS/2;
		if ( xoffset > mouse_x )
		{
			selected_column = i;
			break;
		}
	}
	for (i=1; i<rows+1; i++)
	{
		yoffset = i*colwidth + TOP_MARGINS/2;
		if ( yoffset > mouse_y )
		{
			selected_row = i;
			break;
		}
	}

	/*
	 * Ignore selections outside of selectable regions.
	 */
	if ( ( selected_row == 0 ) || ( selected_column == 0 ) )
		return;

	/*
	 * Compute the new arrowhead index.
	 */
	new_index = (selected_row-1) * ARROW_MAX_COLUMNS + selected_column;

	/*
	 * If we selected the same arrowhead that was previously selected then
	 * don't do anything.  Similarly, if we select an arrowhead that doesn't
	 * exist, don't do anything.
	 */
	if ( new_index == selected_index )
		return;
	if ( new_index > ARROW_DEFS )
		return;

	/*
	 * Clear the old selection by whiting-out the selection box.
	 */
	if (selected_index != 0 )
	{
		row = (selected_index-1) / ARROW_MAX_COLUMNS;
		col = (selected_index-1) % ARROW_MAX_COLUMNS;

		xoffset = col * colwidth + SIDE_MARGINS/2;
		yoffset = row * rowwidth + TOP_MARGINS/2;
		gdk_draw_rectangle(
			darea->widget.window, gcWhite, FALSE,
			xoffset, yoffset, colwidth, rowwidth);

	}

	/*
	 * Draw black rectangle around new selected arrowhead.
	 */
	xoffset = (selected_column - 1) * colwidth + SIDE_MARGINS/2;
	yoffset = (selected_row - 1) * rowwidth + TOP_MARGINS/2;

	gdk_draw_rectangle(
			darea->widget.window, gcBlack, FALSE,
			xoffset, yoffset, colwidth, rowwidth);

	/*
	 * Save new index id.
	 */
	selected_index = new_index;
	DBGPrintf(DBG_INFO,
		("new index: %d \t selected_index: %d\n", new_index, selected_index));

}

