/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: arrows.c
 *            DESCRIPTION: main module for ArrowGFX plug-in
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define MAIN_C		/* needed by debug library */

/* === System Headers === */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <math.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "arrows.h"
#include "arrowheads.h"
#include "gfxcommon.h"
#include "gm-logo.xpm"
#include "debug.h"


/*
 * _GimpPlugInInfo is a structure that tells the gimp what routines to
 * run at certain times.  For most plug-ins, init_proc() and 
 * quit_proc() are unnecessary.  All plug-ins will want to define
 * a query_proc() and a run_proc().
typedef struct _GimpPlugInInfo GPlugInfo;
 */


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();

extern void GFXDrawArrowHeads();
extern void GFXPreviewButtonPress();
extern void GFXPreviewButtonRelease();
extern void GFXPreviewMotion();
extern void GFXPreviewConfigure();
extern void GFXTypeToggle();
extern void GFXSizeSelect();
extern void GFXArrowCallback();
extern void GFXTextEntryUpdates();
extern void GFXDPITextUpdate();
extern void GFXDPISliderUpdate();
extern void GFXScalingTextUpdate();
extern void GFXScalingSliderUpdate();
extern void GFXTailTextUpdate();
extern void GFXTailSliderUpdate();

extern gint GFXImageConstraints();
extern void GFXImageSelect();
extern void GFXBlendSelect();
extern void GFXOpacityTextUpdate();
extern void GFXOpacitySliderUpdate();
extern void GFXRebuildImageMenu();
extern void GFXOffsetXTextUpdate();
extern void GFXOffsetXSliderUpdate();
extern void GFXOffsetYTextUpdate();
extern void GFXOffsetYSliderUpdate();
extern void GFXMarginXTextUpdate();
extern void GFXMarginXSliderUpdate();
extern void GFXMarginYTextUpdate();
extern void GFXMarginYSliderUpdate();

extern void GFXToolToggle();
extern void GFXFillToggle();
extern void GFXTailToggle();
extern void GFXAirbrushTextUpdate();
extern void GFXAirbrushSliderUpdate();
extern void GFXPaintbrushTextUpdate();
extern void GFXPaintbrushSliderUpdate();


/* === Public routine prototypes === */
static void ArrowGFXQuery();	/* static, but made global with PLUG_IN_INFO */
static void ArrowGFXRun();		/* static, but made global with PLUG_IN_INFO */
void GFXPreviewUpdate();


/* === Private routine prototypes === */
static void ArrowGFX();		/* the heart of the plug-in */
static void CloseCallback();
static void OKCallback();
static void ResetFields();
static void ShowStatus();
static void ShowHelp();


/* === Global Variables === */
extern int selected_index;
extern int size_type;

GimpPlugInInfo PLUG_IN_INFO = {	/* Required:  you *must* name this PLUG_IN_INFO */
	NULL,				/* No Init procedure */
	NULL,				/* No Exit procedure */
	ArrowGFXQuery,	/* Query procedure */
	ArrowGFXRun,	/* Run time procedure - where the work really starts */
};

GtkWidget		*arrowgfx_dialog;
GtkDrawingArea	*preview;
GtkWidget		*tail_length_text;
GtkWidget		*tail_length_slider;
GtkObject		*tail_length_adj;
GtkWidget		*scaling_text;
GtkWidget		*scaling_slider;
GtkObject		*scaling_adj;
GtkWidget		*dpi_text;
GtkWidget		*dpi_slider;
GtkObject		*dpi_adj;
GtkWidget		*opacity_text;
GtkWidget		*opacity_slider;
GtkObject		*opacity_adj;
GtkWidget		*airbrush_text;
GtkWidget		*airbrush_slider;
GtkObject		*airbrush_adj;
GtkWidget		*paintbrush_text;
GtkWidget		*paintbrush_slider;
GtkObject		*paintbrush_adj;
GtkWidget		*offset_x_text;
GtkWidget		*offset_x_slider;
GtkObject		*offset_x_adj;
GtkWidget		*offset_y_text;
GtkWidget		*offset_y_slider;
GtkObject		*offset_y_adj;
GtkWidget		*margin_x_text;
GtkWidget		*margin_x_slider;
GtkObject		*margin_x_adj;
GtkWidget		*margin_y_text;
GtkWidget		*margin_y_slider;
GtkObject		*margin_y_adj;
GtkWidget		*size_menu;
GtkWidget		*size_label;
GtkWidget		*new_layer_radio_button;
GtkWidget		*pencil_radio_button;
GtkWidget		*fill_radio_button;
GtkWidget		*tail_radio_button;
GtkWidget		*angle_text;
GtkWidget		*blend_menu;
GtkWidget		*image_options;
GtkWidget		*image_menu;

GtkWidget		*msg_widget;
GtkWidget		*msg_data;
GtkWidget		*help_widget=NULL;
GtkWidget		*status_label;

GimpDrawable		*drawable;
gint32			drawable_id;

int		mouse_x, mouse_y;		/* current location of mouse on mouse clicks */
int		image_id;				/* ID of image initial layer lives in */
int		blend_mode = 0;		/* NORMAL mode by default for new layers */

GdkPoint	endpoint;				/* XY coords for corners of page in preview */
int		active_type;			/* new layer or anchor to original? */
int		tool_type;				/* pencil, paintbrush, etc. */
int		fill_type;				/* fill, outline, or pattern */
int		tail_status;			/* do or don't draw the tail */
gfloat	rotation_angle=0;		/* current rotation amount */
int		arrow_rotate;			/* if True: mouse drags rotate page */


static char *size_strings[] = {
	"Inches",
	"Pixels",
	NULL
};

static char *blend_strings[] = {
	"Normal",
	"Dissolve",
	"Behind",
	"Multiply",
	"Screen",
	"Overlay",
	"Difference",
	"Addition",
	"Subtract",
	"Darken Only",
	"Lighten Only",
	"Hue",
	"Saturation",
	"Color",
	"Value",
	NULL
};


/*========================================================================
 *	Name:			ArrowGFXQuery
 *	Prototype:	static void ArrowGFXQuery()
 *					
 *	Description:
 *		Register this plug-in in the Gimp Procedural Database.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *	Most of the gimp_install_procedure() arguments are defined values that
 * can be found in the arrows.h file.
 *========================================================================*/
static void
ArrowGFXQuery()
{
	static GimpParamDef args[] = {
		{ GIMP_PDB_INT32,		"run_mode",	"Interactive only" },
		{ GIMP_PDB_IMAGE,		"image",		"Input image" },
		{ GIMP_PDB_DRAWABLE,	"drawable",	"Input drawable" },
		{ GIMP_PDB_LAYER,		"layer",		"Input layer" },
		{ GIMP_PDB_INT32,		"angle",		"Angle of rotation in degrees" },
		{ GIMP_PDB_INT32,		"type",		"0: Create new layer 1: Anchor to original" },
		{ GIMP_PDB_INT32,		"arrowhead","Arrow head ID" },
		{ GIMP_PDB_INT32,		"length",	"length of Arrow" },
		{ GIMP_PDB_INT32,		"dpi",		"DPI, aka resolution" },
		{ GIMP_PDB_INT32,		"units",		"0:pixels  1:inches" },
	};
	static int	nargs = sizeof(args) / sizeof(args[0]);

	gimp_install_procedure(
		ARROWGFX_NAME,
		ARROWGFX_BLURB,
		ARROWGFX_HELP,
		ARROWGFX_AUTHOR,
		ARROWGFX_COPYRIGHT,
		ARROWGFX_DATE,
		ARROWGFX_MENU_PATH,
		ARROWGFX_IMAGE_TYPES,
		ARROWGFX_PROC_TYPE,
		nargs,
		0,
		args,
		NULL
	);
}


/*========================================================================
 *	Name:			ArrowGFXRun
 *	Prototype:	static void ArrowGFXRun()
 *					
 *	Description:
 *		Starts the plug-in.  Basically just preps for creatingg the dialog
 *		and does some initialization before calling the heart of the 
 *		plugin, ArrowGFX().
 *
 *	Input Arguments:
 *	char		*name				Name of print program.
 *	int		nparams			Number of parameters passed in
 *	GimpParam	*param			Parameter values
 *
 *	Output Arguments:
 *	int		*nreturn_vals	Number of return values
 *	GimpParam	**return_vals	Return values
 *
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *========================================================================*/
static void
ArrowGFXRun(
	char		*name,			/* Name of print program. */
	int		nparams,			/* Number of parameters passed in */
	GimpParam	*param,			/* Parameter values */
	int		*nreturn_vals,	/* Number of return values */
	GimpParam	**return_vals	/* Return values */
)
{
	GimpRunModeType	run_mode;
	GimpParam			*values;

	/*
	 * Initialize parameter data...
	 */
	run_mode = param[0].data.d_int32;
	values = g_new(GimpParam, 1);
	values[0].type          = GIMP_PDB_STATUS;
	values[0].data.d_status = GIMP_PDB_SUCCESS;
	*nreturn_vals = 1;
	*return_vals  = values;
 
	/*
	 * Get drawable.  This is what we will be scaling and duplicating
	 * unless the user selects another one.
	 */
	drawable = gimp_drawable_get(param[2].data.d_drawable);
	drawable_id = param[2].data.d_drawable;

	image_id = gimp_drawable_image_id(drawable_id);


	/*
	 * Do run-mode specific setup.
	 */
	switch(run_mode)
	{
		case GIMP_RUN_INTERACTIVE:
			/*
			gimp_get_data(ARROWGFX_PI_NAME, &init_values);
			*/

			ArrowGFX();
			break;

		case GIMP_RUN_NONINTERACTIVE:
			break;

		case GIMP_RUN_WITH_LAST_VALS:
			break;

		default:
			values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
			break;
	}
}



/*========================================================================
 *	Name:			MAIN
 *	Prototype:	MAIN()
 *					
 *	Description:
 *		Gimp macro that defines a plug-in's main routine.  All plug-ins must
 *		include this.
 *========================================================================*/
MAIN();



/*========================================================================
 *	Name:			ArrowGFX
 *	Prototype:	static void ArrowGFX()
 *					
 *	Description:
 *		Set up the dialog and display it.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ArrowGFX()
{
#ifdef DEBUG
	char			*fname="ArrowGFX()";
#endif

	gint			argc;
	gchar			**argv;
	char			buf[256];
	int			index;

	GtkWidget	*notebook, *toptable;
	GtkWidget	*table;
	GtkWidget	*hbox, *vbox, *frame, *hsep;
	GtkWidget	*label;
	GtkWidget	*label_pixmap;
	GtkWidget	*arrow;
	GtkWidget	*option_menu, *menu, *menuitem;
	GtkWidget	*toggle;
	GtkWidget	*reset_button;
	GtkWidget	*ok_button, *cancel_button, *help_button;
/*
	GtkWidget	*status_button;
*/
	GtkWidget	*angle_up_button;
	GtkWidget	*angle_down_button;
	GdkPixmap	*logo;
	GdkBitmap	*logo_mask;
	GtkStyle		*style;

	GSList		*group = NULL;

	char			dbgfile[256];

	/*
	 * Generic initialization:
	 * 0. Fake the argc/argv stuff.
	 * 1. Initialize the GTK toolkit.
	 * 2. parse the Gimp's gtkrc for plug-in specific settings.
	 * 3. Set up to use XSharedMemory, if possible.
	 */
	argc    = 1;
	argv    = g_new(gchar *, 1);
	argv[0] = g_strdup("bcards");
	gtk_init(&argc, &argv);
	gtk_rc_parse(gimp_gtkrc());
	gdk_set_use_xshm(gimp_use_xshm());

	/*
	 * Set up for debugging.
	 */
#ifdef DEBUG
	sprintf(dbgfile,"/tmp/gfxarrow");
	DBGFile=dbgfile;
	GMDebugLevel=DBGStringToInt("0xffff");
	DBGOpen( DEBUG_FD );
	DBGEnter();
#endif

	/*
	 * Try to behave in a civil manner should the unexpected occur.
	 */
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);

	/*
	 * Create a new dialog window.
	 */
	arrowgfx_dialog = gtk_dialog_new();
	sprintf(buf, "%s - %s", ARROWGFX_TITLE, ARROWGFX_VERSION);
	gtk_window_set_title(GTK_WINDOW(arrowgfx_dialog), buf);
	gtk_window_set_wmclass(GTK_WINDOW(arrowgfx_dialog), "arrowgfx", "Gimp");
	gtk_window_position(GTK_WINDOW(arrowgfx_dialog), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(arrowgfx_dialog), 0);
	gtk_signal_connect(GTK_OBJECT(arrowgfx_dialog), "destroy",
		(GtkSignalFunc)CloseCallback, NULL);

	gtk_widget_realize(arrowgfx_dialog);

	/*
	 * Initialize the pixmap.
	 */
	style = gtk_widget_get_style( arrowgfx_dialog );
	logo = gdk_pixmap_create_from_xpm_d(
					arrowgfx_dialog->window, 
					&logo_mask, 
					&style->bg[GTK_STATE_NORMAL], 
					gm_logo_xpm);

	/*
	 * The main table in which everything else will go.
	 */
	toptable = gtk_table_new(3, 1, FALSE);
	gtk_container_border_width(GTK_CONTAINER(toptable), 1);
	gtk_table_set_col_spacings(GTK_TABLE(toptable), 1);
	gtk_table_set_row_spacings(GTK_TABLE(toptable), 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (arrowgfx_dialog)->action_area), 
		toptable, TRUE, TRUE, 0);
	gtk_widget_show(toptable);

	/*
	 * Main Notebook with 3 tabs.
	 */
	notebook = gtk_notebook_new();
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)notebook, 
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_widget_show(notebook);


	/*
	 * The table for the first page.
	 */
	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 1);
	gtk_table_set_col_spacings(GTK_TABLE(table), 1);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_widget_show(table);

	/*
	 * Add the table to page 1 of the notebook.
	 */
	label = gtk_label_new("Arrow Options");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, label);

	/*
	 * A small preview window that will be used for rotating interactively.
	 */
	frame = gtk_frame_new("Rotation Angle Preview");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	preview = (GtkDrawingArea *)gtk_drawing_area_new();
	gtk_drawing_area_size(preview, PREVIEW_WSIZE, PREVIEW_HSIZE);
	gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *)preview);
	gtk_widget_show((GtkWidget *)preview);

	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "expose_event",
			(GtkSignalFunc)GFXPreviewUpdate,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_press_event",
			(GtkSignalFunc)GFXPreviewButtonPress,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_release_event",
			(GtkSignalFunc)GFXPreviewButtonRelease,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "motion_notify_event",
			(GtkSignalFunc)GFXPreviewMotion,
			NULL);

	gtk_widget_set_events((GtkWidget *)preview,
		GDK_EXPOSURE_MASK | GDK_BUTTON_MOTION_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
 
	/*
	 * Arrow head types.
	 */
	frame = gtk_frame_new("Arrow Heads");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 3, 5, 0, 0, 0, 0);
	gtk_widget_show(frame);

	GFXDrawArrowHeads(frame);

	/*
	 * The Muse Logo.
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);
	label_pixmap = gtk_pixmap_new(logo, logo_mask);
	gtk_container_add (GTK_CONTAINER (frame), label_pixmap);
	gtk_widget_show(label_pixmap);

	/*
	 * Angle in degrees input field and arrow buttons.
	 */
	frame = gtk_frame_new("Angle (in degrees)");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_show(vbox);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	gtk_widget_show(hbox);

	/* Arrow Up */
	angle_down_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (angle_down_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (angle_down_button), "clicked",
		(GtkSignalFunc) GFXArrowCallback, (gpointer)ARROWGFX_DOWN);
	gtk_box_pack_start (GTK_BOX (hbox), angle_down_button, TRUE, TRUE, 4);
	gtk_widget_show(angle_down_button);

	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (angle_down_button), arrow);
	gtk_widget_show(arrow);

	/* Input field */
	angle_text = gtk_entry_new_with_max_length(7);
	gtk_widget_set_usize( angle_text, 70, -1 );
	gtk_box_pack_start (GTK_BOX (hbox), angle_text, TRUE, TRUE, 4);
	gtk_widget_show(angle_text);
	sprintf(buf, "%.3f", (gfloat)rotation_angle);
	gtk_entry_set_text(GTK_ENTRY(angle_text), buf);
	gtk_signal_connect (GTK_OBJECT (angle_text), "changed",
		(GtkSignalFunc) GFXTextEntryUpdates, (gpointer)NULL);

	/* Arrow Down */
	angle_up_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (angle_up_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (angle_up_button), "clicked",
		(GtkSignalFunc) GFXArrowCallback, (gpointer)ARROWGFX_UP);
	gtk_box_pack_start (GTK_BOX (hbox), angle_up_button, TRUE, TRUE, 4);
	gtk_widget_show(angle_up_button);

	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (angle_up_button), arrow);
	gtk_widget_show(arrow);

	/*
	 * Tail length slider.
	 */
	frame = gtk_frame_new("Tail Length");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_container_add (GTK_CONTAINER (vbox), frame);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	tail_length_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), tail_length_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( tail_length_text, 50, -1 );
	sprintf(buf, "0");
	gtk_entry_set_text(GTK_ENTRY(tail_length_text), buf);
	gtk_signal_connect(GTK_OBJECT(tail_length_text), "changed",
		(GtkSignalFunc)GFXTailTextUpdate, NULL);
	gtk_widget_show(tail_length_text);

	tail_length_adj = 
		gtk_adjustment_new(0, 0.0, 500.0, 1.0, 10.0, 10.0);
	tail_length_slider = gtk_hscale_new(GTK_ADJUSTMENT(tail_length_adj));
	gtk_box_pack_start (GTK_BOX (hbox), tail_length_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( tail_length_slider, 135, -1 );
	gtk_widget_show(tail_length_slider);

	gtk_signal_connect(GTK_OBJECT(tail_length_adj), "value_changed",
		(GtkSignalFunc)GFXTailSliderUpdate, NULL);


	/*
	 * Scaling slider.
	 */
	frame = gtk_frame_new("Scaling");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	scaling_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), scaling_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( scaling_text, 50, -1 );
	sprintf(buf, "%.1f", SCALING_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(scaling_text), buf);
	gtk_signal_connect(GTK_OBJECT(scaling_text), "changed",
		(GtkSignalFunc)GFXScalingTextUpdate, NULL);
	gtk_widget_show(scaling_text);

	scaling_adj = 
		gtk_adjustment_new(SCALING_DEFAULT, 0.0, 10.0, 0.1, 1.0, 1.0);
	scaling_slider = gtk_hscale_new(GTK_ADJUSTMENT(scaling_adj));
	gtk_box_pack_start (GTK_BOX (hbox), scaling_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( scaling_slider, 135, -1 );
	gtk_widget_show(scaling_slider);

	gtk_signal_connect(GTK_OBJECT(scaling_adj), "value_changed",
		(GtkSignalFunc)GFXScalingSliderUpdate, NULL);

	/*
	 * DPI slider.
	 */
	frame = gtk_frame_new("DPI");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 3, 4, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	dpi_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), dpi_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( dpi_text, 50, -1 );
	sprintf(buf, "%d", DPI_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	gtk_signal_connect(GTK_OBJECT(dpi_text), "changed",
		(GtkSignalFunc)GFXDPITextUpdate, NULL);
	gtk_widget_show(dpi_text);

	dpi_adj = 
		gtk_adjustment_new(DPI_DEFAULT, 50.0, 1500.0, 1.0, 10.0, 10.0);
	dpi_slider = gtk_hscale_new(GTK_ADJUSTMENT(dpi_adj));
	gtk_box_pack_start (GTK_BOX (hbox), dpi_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( dpi_slider, 135, -1 );
	gtk_widget_show(dpi_slider);

	gtk_signal_connect(GTK_OBJECT(dpi_adj), "value_changed",
		(GtkSignalFunc)GFXDPISliderUpdate, NULL);

	/*
	 * Current size display.
	 */
	frame = gtk_frame_new("Size");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 4, 5, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

/*
	sprintf(buf, "%.3f x %.3f", SCALING_DEFAULT, SCALING_DEFAULT);
*/
	sprintf(buf, "             ");
	size_label = label = gtk_label_new(buf);
	gtk_widget_set_usize( label, 105, -1 );
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	size_menu = option_menu = gtk_option_menu_new();
	gtk_widget_set_usize( option_menu, 80, -1 );
	gtk_box_pack_start (GTK_BOX (hbox), option_menu, TRUE, TRUE, 0);
	gtk_widget_show(option_menu);

	menu = gtk_menu_new();
	index=0;
   while ( size_strings[index] != NULL )
   {
      menuitem = gtk_menu_item_new_with_label(size_strings[index]);
      gtk_menu_append(GTK_MENU(menu), menuitem);
      gtk_widget_show(menuitem);
      gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
         GTK_SIGNAL_FUNC(GFXSizeSelect), (gpointer) index);
      index++;
   }  
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);


	/*
	 * Page 2:
	 * Another table to line things up nicely.
	 */
	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 8);
	gtk_widget_show(table);

	/*
	 * Add the table to page 2 of the notebook.
	 */
	label = gtk_label_new("Image/Layer Options");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, label);

	/*
	 * Toggle buttons for Layer/Mask/Channel dispositions.
	 */
	frame = gtk_frame_new("Draw to:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_set_usize( vbox, 165, -1 );
	gtk_widget_show(vbox);

	new_layer_radio_button = toggle = 
		gtk_radio_button_new_with_label(NULL, "New Layer");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	active_type = ARROWGFX_TYPE_NEW_LAYER;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTypeToggle, (gpointer) ARROWGFX_TYPE_NEW_LAYER);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "To Original");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTypeToggle, (gpointer) ARROWGFX_TYPE_ANCHOR);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "To Mask");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTypeToggle, (gpointer) ARROWGFX_TYPE_MASK);
	gtk_widget_show (toggle);

/*
 * Future development
	toggle = gtk_radio_button_new_with_label(group, "To Channel");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTypeToggle, (gpointer) ARROWGFX_TYPE_CHANNEL);
	gtk_widget_show (toggle);
*/

	/*
	 * Offset margins for arrow within layer/mask.
	 */
	frame = gtk_frame_new("Drawing Margins:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 2, 4, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_set_usize( vbox, 165, -1 );
	gtk_widget_show(vbox);

	frame = gtk_frame_new("X Margin");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	margin_x_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), margin_x_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( margin_x_text, 50, -1 );
	sprintf(buf, "%d", MARGIN_X_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(margin_x_text), buf);
	gtk_signal_connect(GTK_OBJECT(margin_x_text), "changed",
		(GtkSignalFunc)GFXMarginXTextUpdate, NULL);
	gtk_widget_show(margin_x_text);

	margin_x_adj = 
		gtk_adjustment_new(MARGIN_X_DEFAULT, 0.0, 60.0, 1.0, 10.0, 10.0);
	margin_x_slider = gtk_hscale_new(GTK_ADJUSTMENT(margin_x_adj));
	gtk_box_pack_start (GTK_BOX (hbox), margin_x_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( margin_x_slider, 108, -1 );
	gtk_widget_show(margin_x_slider);

	gtk_signal_connect(GTK_OBJECT(margin_x_adj), "value_changed",
		(GtkSignalFunc)GFXMarginXSliderUpdate, NULL);

	frame = gtk_frame_new("Y Margin");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	margin_y_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), margin_y_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( margin_y_text, 50, -1 );
	sprintf(buf, "%d", MARGIN_Y_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(margin_y_text), buf);
	gtk_signal_connect(GTK_OBJECT(margin_y_text), "changed",
		(GtkSignalFunc)GFXMarginYTextUpdate, NULL);
	gtk_widget_show(margin_y_text);

	margin_y_adj = 
		gtk_adjustment_new(MARGIN_Y_DEFAULT, 0.0, 60.0, 1.0, 10.0, 10.0);
	margin_y_slider = gtk_hscale_new(GTK_ADJUSTMENT(margin_y_adj));
	gtk_box_pack_start (GTK_BOX (hbox), margin_y_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( margin_y_slider, 108, -1 );
	gtk_widget_show(margin_y_slider);

	gtk_signal_connect(GTK_OBJECT(margin_y_adj), "value_changed",
		(GtkSignalFunc)GFXMarginYSliderUpdate, NULL);


	/*
	 * The menu of Images currently available.
	 */
	frame = gtk_frame_new("Available images:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);

	image_options = option_menu = gtk_option_menu_new();
	gtk_widget_set_usize( option_menu, 185, -1 );
	gtk_container_add (GTK_CONTAINER (frame), image_options);
	gtk_widget_show(option_menu);

	image_menu = menu = gimp_image_menu_new(GFXImageConstraints,
					GFXImageSelect,
					NULL,
					image_id);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);

	/*
	 * Layer Blend menu and Opacity slider.
	 * Sets default blend mode for new layers.
	 */
	frame = gtk_frame_new("New Layer Options");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 1, 4, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_show(vbox);

	frame = gtk_frame_new("Blend mode:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	blend_menu = option_menu = gtk_option_menu_new();
	gtk_widget_set_usize( option_menu, 185, -1 );
	gtk_container_add (GTK_CONTAINER (frame), option_menu);
	gtk_widget_show(option_menu);

	menu = gtk_menu_new();
	index=0;
   while ( blend_strings[index] != NULL )
   {
      menuitem = gtk_menu_item_new_with_label(blend_strings[index]);
      gtk_menu_append(GTK_MENU(menu), menuitem);
      gtk_widget_show(menuitem);
      gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
         GTK_SIGNAL_FUNC(GFXBlendSelect), (gpointer) index);
      index++;
   }  
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);


	/*
	 * Opacity slider.
	 */
	frame = gtk_frame_new("Opacity");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	opacity_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), opacity_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( opacity_text, 50, -1 );
	sprintf(buf, "%.1f", OPACITY_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(opacity_text), buf);
	gtk_signal_connect(GTK_OBJECT(opacity_text), "changed",
		(GtkSignalFunc)GFXOpacityTextUpdate, NULL);
	gtk_widget_show(opacity_text);

	opacity_adj = 
		gtk_adjustment_new(OPACITY_DEFAULT, 0.0, 110.0, 1.0, 10.0, 10.0);
	opacity_slider = gtk_hscale_new(GTK_ADJUSTMENT(opacity_adj));
	gtk_box_pack_start (GTK_BOX (hbox), opacity_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( opacity_slider, 135, -1 );
	gtk_widget_show(opacity_slider);

	gtk_signal_connect(GTK_OBJECT(opacity_adj), "value_changed",
		(GtkSignalFunc)GFXOpacitySliderUpdate, NULL);

	/*
	 * Offset options - how far to offset the arrow from the upper left
	 * corner of the image window for new layers.
	 */
	frame = gtk_frame_new("X Offset");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	offset_x_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), offset_x_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( offset_x_text, 50, -1 );
	sprintf(buf, "%d", OFFSET_X_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(offset_x_text), buf);
	gtk_signal_connect(GTK_OBJECT(offset_x_text), "changed",
		(GtkSignalFunc)GFXOffsetXTextUpdate, NULL);
	gtk_widget_show(offset_x_text);

	offset_x_adj = 
		gtk_adjustment_new(OFFSET_X_DEFAULT, 0.0, 1010.0, 1.0, 10.0, 10.0);
	offset_x_slider = gtk_hscale_new(GTK_ADJUSTMENT(offset_x_adj));
	gtk_box_pack_start (GTK_BOX (hbox), offset_x_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( offset_x_slider, 135, -1 );
	gtk_widget_show(offset_x_slider);

	gtk_signal_connect(GTK_OBJECT(offset_x_adj), "value_changed",
		(GtkSignalFunc)GFXOffsetXSliderUpdate, NULL);

	frame = gtk_frame_new("Y Offset");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	gtk_widget_show(hbox);

	offset_y_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), offset_y_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( offset_y_text, 50, -1 );
	sprintf(buf, "%d", OFFSET_Y_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(offset_y_text), buf);
	gtk_signal_connect(GTK_OBJECT(offset_y_text), "changed",
		(GtkSignalFunc)GFXOffsetYTextUpdate, NULL);
	gtk_widget_show(offset_y_text);

	offset_y_adj = 
		gtk_adjustment_new(OFFSET_Y_DEFAULT, 0.0, 1010.0, 1.0, 10.0, 10.0);
	offset_y_slider = gtk_hscale_new(GTK_ADJUSTMENT(offset_y_adj));
	gtk_box_pack_start (GTK_BOX (hbox), offset_y_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( offset_y_slider, 135, -1 );
	gtk_widget_show(offset_y_slider);

	gtk_signal_connect(GTK_OBJECT(offset_y_adj), "value_changed",
		(GtkSignalFunc)GFXOffsetYSliderUpdate, NULL);


	/*
	 * Page 3: Paint/Draw Options
	 * Another table to line things up nicely.
	 */
	table = gtk_table_new(3, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 8);
	gtk_widget_show(table);

	/*
	 * Add the table to page 2 of the notebook.
	 */
	label = gtk_label_new("Draw Options");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, label);

	/*
	 * Options for type of drawing to use.
	 */
	frame = gtk_frame_new("Draw With:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_set_usize( vbox, 165, -1 );
	gtk_widget_show(vbox);

	pencil_radio_button = toggle = 
		gtk_radio_button_new_with_label(NULL, "Pencil");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	tool_type = ARROWGFX_TYPE_PENCIL;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXToolToggle, (gpointer) ARROWGFX_TYPE_PENCIL);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Paintbrush");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXToolToggle, (gpointer) ARROWGFX_TYPE_PAINTBRUSH);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Airbrush");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXToolToggle, (gpointer) ARROWGFX_TYPE_AIRBRUSH);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Stroke Outline");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXToolToggle, (gpointer) ARROWGFX_TYPE_STROKE);
	gtk_widget_show (toggle);

	/*
	 * Tool type specific options.
	 */
	frame = gtk_frame_new("Paintbrush Fade Out");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	paintbrush_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), paintbrush_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( paintbrush_text, 50, -1 );
	sprintf(buf, "%d", PAINTBRUSH_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(paintbrush_text), buf);
	gtk_signal_connect(GTK_OBJECT(paintbrush_text), "changed",
		(GtkSignalFunc)GFXPaintbrushTextUpdate, NULL);
	gtk_widget_show(paintbrush_text);

	paintbrush_adj = 
		gtk_adjustment_new(PAINTBRUSH_DEFAULT, 0.0, 1010.0, 1.0, 10.0, 10.0);
	paintbrush_slider = gtk_hscale_new(GTK_ADJUSTMENT(paintbrush_adj));
	gtk_box_pack_start (GTK_BOX (hbox), paintbrush_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( paintbrush_slider, 135, -1 );
	gtk_widget_show(paintbrush_slider);

	gtk_signal_connect(GTK_OBJECT(paintbrush_adj), "value_changed",
		(GtkSignalFunc)GFXPaintbrushSliderUpdate, NULL);

	frame = gtk_frame_new("Airbrush Pressure");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	airbrush_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), airbrush_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( airbrush_text, 50, -1 );
	sprintf(buf, "%d", AIRBRUSH_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(airbrush_text), buf);
	gtk_signal_connect(GTK_OBJECT(airbrush_text), "changed",
		(GtkSignalFunc)GFXAirbrushTextUpdate, NULL);
	gtk_widget_show(airbrush_text);

	airbrush_adj = 
		gtk_adjustment_new(AIRBRUSH_DEFAULT, 0.0, 110.0, 1.0, 10.0, 10.0);
	airbrush_slider = gtk_hscale_new(GTK_ADJUSTMENT(airbrush_adj));
	gtk_box_pack_start (GTK_BOX (hbox), airbrush_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( airbrush_slider, 135, -1 );
	gtk_widget_show(airbrush_slider);

	gtk_signal_connect(GTK_OBJECT(airbrush_adj), "value_changed",
		(GtkSignalFunc)GFXAirbrushSliderUpdate, NULL);


	/* Set appropriate sensitivity for these toggles */
	GFXToolToggle(toggle, tool_type);


	/*
	 * Options for fill type to use.
	 */
	frame = gtk_frame_new("Fill Type:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_set_usize( vbox, 165, -1 );
	gtk_widget_show(vbox);

	fill_radio_button = toggle = 
		gtk_radio_button_new_with_label(NULL, "Solid Background Color");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	fill_type = ARROWGFX_TYPE_FILL;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXFillToggle, (gpointer) ARROWGFX_TYPE_FILL);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Pattern With Outline");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXFillToggle, (gpointer) ARROWGFX_TYPE_PATTERN_OL);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Pattern, No Outline");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXFillToggle, (gpointer) ARROWGFX_TYPE_PATTERN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Outline only");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXFillToggle, (gpointer) ARROWGFX_TYPE_OUTLINE);
	gtk_widget_show (toggle);

	/*
	 * Tail options.
	 */
	frame = gtk_frame_new("Tail:");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_set_usize( vbox, 165, -1 );
	gtk_widget_show(vbox);

	tail_radio_button = toggle = 
		gtk_radio_button_new_with_label(NULL, "Draw");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	tail_status = ARROWGFX_TYPE_DRAW;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTailToggle, (gpointer) ARROWGFX_TYPE_DRAW);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Don't Draw");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) GFXTailToggle, (gpointer) ARROWGFX_TYPE_DONT_DRAW);
	gtk_widget_show (toggle);


	/*
	 * Horizontal separator between main window area and status buttons.
	 */
	hsep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)hsep, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(hsep);


	/*
	 * Buttons go along the bottom row of the table.
	 */
	hbox = gtk_hbox_new(TRUE, 2);
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)hbox, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(hbox);

	/*
	 * Ok Button
	 */
	ok_button = gtk_button_new_with_label ("OK");
	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
		(GtkSignalFunc) OKCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), ok_button, TRUE, TRUE, 2);
	gtk_widget_grab_default (ok_button);
	gtk_widget_show(ok_button);

	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (ok_button);

	/*
	 * Cancel Button
	 */
	cancel_button = gtk_button_new_with_label ("Cancel");
	GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		(GtkSignalFunc) CloseCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), cancel_button, TRUE, TRUE, 2);
	gtk_widget_show(cancel_button);

	/*
	 * Reset Button
	 */
	reset_button = gtk_button_new_with_label("Reset");
	GTK_WIDGET_SET_FLAGS (reset_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(reset_button), "clicked",
		(GtkSignalFunc)ResetFields, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), reset_button, TRUE, TRUE, 2);
	gtk_widget_show(reset_button);

	/*
	 * Status Button
	status_button = gtk_button_new();

	status_label = gtk_label_new("Show Status");
	gtk_container_add (GTK_CONTAINER (status_button), (GtkWidget *)status_label);
	gtk_widget_show(status_label);

	GTK_WIDGET_SET_FLAGS (status_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(status_button), "clicked",
		(GtkSignalFunc)ShowStatus, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), status_button, TRUE, TRUE, 2);
	gtk_widget_show(status_button);
	 */

	/*
	 * Help Button
	 */
	help_button = gtk_button_new_with_label ("Help");
	GTK_WIDGET_SET_FLAGS (help_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (hbox), help_button, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(help_button), "clicked",
		(GtkSignalFunc)ShowHelp, NULL);
	gtk_widget_show(help_button);

	/*
	 * Show the main dialog.
	 */
	gtk_widget_show(arrowgfx_dialog);

	/*
	 * Prevent resizing.
	 */
	gtk_window_set_policy(GTK_WINDOW(arrowgfx_dialog), FALSE, FALSE, FALSE);

	/*
	 * Sit and spin.
	 */
	gtk_main();

}


/*========================================================================
 *	Name:			ShowHelp
 *	Prototype:	static void ShowHelp()
 *					
 *	Description:
 *		Manage the Help dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowHelp(
)
{
	if ( help_widget == NULL )
	{
		GFXMsgWindow(
			GFX_HELP_TEXT,
			GFX_INFO_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&help_widget, NULL, 
			NULL, ShowHelp, NULL, ShowHelp, NULL,
			600, 400);

	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (help_widget) )
		{
			gtk_widget_hide(help_widget);
		}
		else
		{
			gtk_widget_show(help_widget);
			GFXCenterWindow(help_widget);
		}
	}
}



/*========================================================================
 *	Name:			ShowStatus
 *	Prototype:	static void ShowStatus()
 *					
 *	Description:
 *		Manage the status dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowStatus(
)
{
	if ( msg_widget == NULL )
	{
		GFXMsgWindow(
			"ArrowsGFX Status\n---------------------\n",
			GFX_MSG_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&msg_widget, &msg_data, 
			NULL, ShowStatus, NULL, ShowStatus, NULL, 0, 0);

		gtk_label_set_text(GTK_LABEL(status_label), "Hide Status");
	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (msg_widget) )
		{
			gtk_label_set_text(GTK_LABEL(status_label), "Show Status");
			gtk_widget_hide(msg_widget);
		}
		else
		{
			gtk_label_set_text(GTK_LABEL(status_label), "Hide Status");
			gtk_widget_show(msg_widget);
			GFXCenterWindow(msg_widget);
		}
	}
}



/*========================================================================
 *	Name:			CloseCallback
 *	Prototype:	static void CloseCallback()
 *					
 *	Description:
 *		Close the main dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
CloseCallback(
)
{
	gimp_drawable_detach(drawable);
	gtk_main_quit();
}


/*========================================================================
 *	Name:			ResetFields
 *	Prototype:	static void ResetFields()
 *					
 *	Description:
 *		Reset the dialog to its initial configuration.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ResetFields()
{
	char	buf[16];

	gtk_toggle_button_set_state (
			GTK_TOGGLE_BUTTON (new_layer_radio_button), 
			TRUE);

	gtk_toggle_button_set_state (
			GTK_TOGGLE_BUTTON (pencil_radio_button), 
			TRUE);

	gtk_toggle_button_set_state (
			GTK_TOGGLE_BUTTON (fill_radio_button), 
			TRUE);

	gtk_option_menu_set_history (GTK_OPTION_MENU (blend_menu), 0);
	gtk_option_menu_set_history (GTK_OPTION_MENU (size_menu), 0);

	sprintf(buf, "%d", arrow_hdrs[selected_index-1].tail.length);
	gtk_entry_set_text(GTK_ENTRY(tail_length_text), buf);
	sprintf(buf, "%.1f", OPACITY_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(opacity_text), buf);
	sprintf(buf, "%.1f", SCALING_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(scaling_text), buf);
	sprintf(buf, "%d", DPI_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	sprintf(buf, "%d", AIRBRUSH_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(airbrush_text), buf);
	sprintf(buf, "%d", PAINTBRUSH_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(paintbrush_text), buf);
	sprintf(buf, "%d", OFFSET_X_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(offset_x_text), buf);
	sprintf(buf, "%d", OFFSET_Y_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(offset_y_text), buf);
	sprintf(buf, "%d", MARGIN_X_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(margin_x_text), buf);
	sprintf(buf, "%d", MARGIN_Y_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(margin_y_text), buf);

	GFXSizeSelect(0, (gpointer)size_type);

	active_type = ARROWGFX_TYPE_NEW_LAYER;
	tool_type = ARROWGFX_TYPE_PENCIL;
	fill_type = ARROWGFX_TYPE_FILL;
	tail_status = ARROWGFX_TYPE_DRAW;
	blend_mode = 0;

	rotation_angle = 0.0;
	sprintf(buf, "%.3f", rotation_angle);
	gtk_entry_set_text(GTK_ENTRY(angle_text), buf);

	image_id = gimp_drawable_image_id(drawable_id);
	GFXRebuildImageMenu();
	GFXPreviewUpdate();
}


/*========================================================================
 *	Name:			GFXComputePts
 *	Prototype:	void GFXComputePts()
 *					
 *	Description:
 *		Compute current set of pts, bounding box size, etc.
 *
 *	Input Arguments:
 *		gdouble	**fpts		where to return the arrowhead coordinate buffer
 *		gdouble	**tpts		where to return the tail coordinate buffer
 *		int		*bb_width	width of bounding box
 *		int		*bb_height	height of bounding box
 *
 *	Output Arguments:
 *	Return Values:
 *	Global Values:
 *	Method:
 *	Notes:
 *	It is the callers responsibility to free up the buffers allocated for
 *	the fpts and tpts return values.
 *========================================================================*/
void
GFXComputePts(
	gdouble	**fpts,
	gdouble	**tpts,
	int		*bb_width,
	int		*bb_height,
	gfloat	*scale_factor
)
{
	gdouble	*floatpts, *tailpts;
	int		max, min_x, min_y, tail_length, box_height, box_width;
	int		xmargin, ymargin;
	int		i;
	GdkPoint	adj, tail[4];
	gfloat	angle_degrees, angle_radians;
	int		height;
	gfloat	scalar_x, scalar_y;
	int		dpi;
	gfloat	scaling_factor;
	char		buf[32];

	/*
	 * Grab necessary UI values.
 	 */
	dpi = atoi( gtk_entry_get_text(GTK_ENTRY(dpi_text)) );
	angle_degrees = atof( gtk_entry_get_text(GTK_ENTRY(angle_text)) );
	scaling_factor = atof( gtk_entry_get_text(GTK_ENTRY(scaling_text)) );
	xmargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_x_text)) );
	ymargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_y_text)) );

	/*
	 * Validate data.
	 */
	if (scaling_factor > SCALING_MAX_DEFAULT) 
	{
		scaling_factor = SCALING_MAX_DEFAULT;
		sprintf(buf,"%.1f",scaling_factor);
		gtk_entry_set_text(GTK_ENTRY(scaling_text), buf);
	}
	if (dpi > DPI_MAX_DEFAULT) 
	{
		dpi = DPI_MAX_DEFAULT;
		sprintf(buf,"%d", dpi);
		gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	}
	if (xmargin > MARGIN_MAX_DEFAULT) 
	{
		xmargin = MARGIN_MAX_DEFAULT;
		sprintf(buf,"%d", xmargin);
		gtk_entry_set_text(GTK_ENTRY(margin_x_text), buf);
	}
	if (ymargin > MARGIN_MAX_DEFAULT) 
	{
		ymargin = MARGIN_MAX_DEFAULT;
		sprintf(buf,"%d", ymargin);
		gtk_entry_set_text(GTK_ENTRY(margin_y_text), buf);
	}

	/*
	 * Convert degrees into radians.
	 */
	angle_radians = (angle_degrees + 90) * ( 2 * M_PI / 360 );

	/*
	 * Process selected arrowhead points.  The size of the bounding box for
	 * these points has to include the length of tail.  Thats what the next
	 * set of calculations are for.  Eventually, box_height and box_width
	 * are the size of the bounding box.
	 */
	if ( arrow_hdrs[selected_index-1].tail.y1 > 
			arrow_hdrs[selected_index-1].tail.y2 )
		max = arrow_hdrs[selected_index-1].tail.y1;
	else
		max = arrow_hdrs[selected_index-1].tail.y2;

	tail_length = atoi(gtk_entry_get_text(GTK_ENTRY(tail_length_text)));

	if ( (max+tail_length) > arrow_hdrs[selected_index-1].grid_size )
		box_height = max+tail_length;
	else
		box_height = arrow_hdrs[selected_index-1].grid_size;

	box_width = arrow_hdrs[selected_index-1].grid_size;

	/*
	 * Allocate an array of points.
	 */
	floatpts = 
		(gdouble *)g_new(gdouble, (arrow_hdrs[selected_index-1].points+1) * 2 );

	for(i=0; i<arrow_hdrs[selected_index-1].points; i++)
	{
		/*
		 * Image origin is upper left corner and all have their "tips"
		 * at grid_size/2.  So we move the whole thing over by
		 * -grid_size/2 before doing any rotations or other transforms.
		 */
		adj.x = (gint16)arrow_vertices[selected_index-1][i].x - (box_width / 2);
		adj.y = (gint16)arrow_vertices[selected_index-1][i].y - (box_height / 2);

		/*
		 * Rotate image to point in direction of rotation.
		 */
		floatpts[i*2] = 
			(adj.x * cos(angle_radians)) - (adj.y * sin(angle_radians));
		floatpts[i*2+1] = 
			(adj.x * sin(angle_radians)) + (adj.y * cos(angle_radians));

		/*
		 * Reset origin to upper left corner.
		 */
		floatpts[i*2]   += (box_width / 2);
		floatpts[i*2+1] += (box_height / 2);

	}
	/* One more to close loop. */
	floatpts[i*2]   = floatpts[0];
	floatpts[i*2+1] = floatpts[1];

	/*
	 * Computer tail coordinates.
	 */
	tail[0].x = (gint16)arrow_hdrs[selected_index-1].tail.x1;
	tail[0].y = (gint16)arrow_hdrs[selected_index-1].tail.y1;
	tail[1].x = (gint16)arrow_hdrs[selected_index-1].tail.x2;
	tail[1].y = (gint16)arrow_hdrs[selected_index-1].tail.y2;

	height = atoi(gtk_entry_get_text(GTK_ENTRY(tail_length_text)));

	/* 
	 * Other two points have to follow clockwise.
	 */
	tail[2].x = tail[1].x;
	tail[2].y = tail[1].y + height;
	tail[3].x = tail[0].x;
	tail[3].y = tail[0].y + height;

	tailpts = (gdouble *)g_new(gdouble, (5 * 2));

	for (i=0; i<4; i++)
	{
		/* Offset to center on original grid */
		adj.x = (gint16)tail[i].x - (box_width / 2);
		adj.y = (gint16)tail[i].y - (box_height / 2);

		/*
		 * Rotate tail to point in direction of rotation.
		 */
		tailpts[i*2] = 
			(adj.x * cos(angle_radians)) - (adj.y * sin(angle_radians));
		tailpts[i*2+1] = 
			(adj.x * sin(angle_radians)) + (adj.y * cos(angle_radians));

		/*
		 * Realign so upper left of bounding box is the origin.
		 */
		tailpts[i*2] += (box_width / 2);
		tailpts[i*2+1] += (box_height / 2);

	}
	/* One more to close loop. */
	tailpts[i*2]   = tailpts[0];
	tailpts[i*2+1] = tailpts[1];


	/*
	 * Rotations may have caused some coordinates to take on negative
	 * components.  We fix that by finding the lowest coordinates for x and y and
	 * adding the absolute value of that to all points.
	 */
	min_x = min_y = 0;
	for(i=0; i<arrow_hdrs[selected_index-1].points+1; i++)
	{
		if ( ( (int)floatpts[i*2] < 0 ) && ( (int)floatpts[i*2] < min_x ) ) 
			min_x = (int)floatpts[i*2];

		if ( ( (int)floatpts[i*2+1] < 0 ) && ( (int)floatpts[i*2+1] < min_y ) )
			min_y = (int)floatpts[i*2+1];
	}
	for(i=0; i<5; i++)
	{
		if ( ( (int)tailpts[i*2] < 0 ) && ( (int)tailpts[i*2] < min_x ) )
			min_x = (int)tailpts[i*2];

		if ( ( (int)tailpts[i*2+1] < 0 ) && ( (int)tailpts[i*2+1] < min_y ) )
			min_y = (int)tailpts[i*2+1];
	}

	min_x = abs(min_x);
	min_y = abs(min_y);

	for(i=0; i<arrow_hdrs[selected_index-1].points+1; i++)
	{
		floatpts[i*2] += (gfloat) min_x;
		floatpts[i*2+1] += (gfloat) min_y;
	}
	for(i=0; i<5; i++)
	{
		tailpts[i*2] += (gfloat) min_x;
		tailpts[i*2+1] += (gfloat) min_y;
	}

	/*
	 * Now find max x and y values, which will be our new layer sizes.
	 */
	box_width = box_height = 0;
	for(i=0; i<arrow_hdrs[selected_index-1].points+1; i++)
	{
		if ( (int)floatpts[i*2] > box_width ) box_width = floatpts[i*2];
		if ( (int)floatpts[i*2+1] > box_height ) box_height = floatpts[i*2+1];
	}
	for(i=0; i<4; i++)
	{
		if ( (int)tailpts[i*2] > box_width ) box_width = tailpts[i*2];
		if ( (int)tailpts[i*2+1] > box_height ) box_height = tailpts[i*2+1];
	}

	/*
	 * Compute scaling factors.
	 */
	if ( box_width > box_height )
		max = box_width;
	else
		max = box_height;
	scalar_x = (gfloat)dpi / (gfloat)max;
	scalar_y = (gfloat)dpi / (gfloat)max;
	if (scaling_factor != 0 )
	{
			scalar_x *= (gfloat)scaling_factor;
			scalar_y *= (gfloat)scaling_factor;
	}

	/*
	 * Now scale all points to fit in the specified box size, 
	 * plus offset them by the margin amounts.
	 */
	for(i=0; i<arrow_hdrs[selected_index-1].points+1; i++)
	{
		floatpts[i*2]   *= (gdouble)scalar_x;
		floatpts[i*2+1] *= (gdouble)scalar_y;
		floatpts[i*2]   += xmargin;
		floatpts[i*2+1] += ymargin;
	}
	for(i=0; i<5; i++)
	{
		tailpts[i*2]   *= (gdouble)scalar_x;
		tailpts[i*2+1] *= (gdouble)scalar_y;
		tailpts[i*2]   += xmargin;
		tailpts[i*2+1] += ymargin;
	}

	/*
	 * Return computed values to caller.
	 */
	if ( bb_width != NULL ) *bb_width = box_width;
	if ( bb_height != NULL ) *bb_height = box_height;
	if ( fpts != NULL ) *fpts = floatpts;
	if ( tpts != NULL ) *tpts = tailpts;
	if ( scale_factor != NULL ) *scale_factor = scaling_factor;
}



/*========================================================================
 *	Name:			OKCallback
 *	Prototype:	static void OKCallback()
 *					
 *	Description:
 *		Process the image.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *
 *	Global Values:
 * active_type			which disposition type to use (new layer or anchor)
 * blend_mode			mode to use for new layers
 * image_id				which image to use
 *
 *	Method:
 *	1. Gather field data
 *
 *	Notes:
 * We don't do much error checking here.  Lets hope things work out for
 * the best!
 *========================================================================*/
static void
OKCallback()
{
#ifdef DEBUG
	char			*fname="OKCallback()";
#endif

	gfloat	angle_degrees, angle_radians;
	gfloat	pressure, fade;
	gfloat	opacity;
	gint32	*images;
	gint		nimages;
	GimpParam	*return_vals;
	int		nreturn_vals;
	char		buf[256];
	gboolean	found;
	int		i;
	gdouble	*floatpts, *tailpts;
	gint32	active_drawable;
	int		layer_id, mask_id;
	int		base_type;
	gchar		*pixel_block;
	int		dpi;
	int		box_height, box_width;
	int		xoffset, yoffset, xmargin, ymargin;
	gfloat	scaling_factor;

	GimpDrawable	*drawable;
	GimpPixelRgn	pixel_rgn;

	DBGEnter();

	/*
	 * Make sure an arrow type has been selected.
	 */
	if ( selected_index == 0 )
	{
		sprintf(buf, "%s", "You need to select an arrow type first.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);
		return;
	}

	/*
	 * Gather field data not kept in global variables.
	 */
	opacity = atof( gtk_entry_get_text(GTK_ENTRY(opacity_text)) );
	angle_degrees = atof( gtk_entry_get_text(GTK_ENTRY(angle_text)) );
	scaling_factor = atof( gtk_entry_get_text(GTK_ENTRY(scaling_text)) );
	dpi = atoi( gtk_entry_get_text(GTK_ENTRY(dpi_text)) );
	fade = atof( gtk_entry_get_text(GTK_ENTRY(paintbrush_text)) );
	pressure = atof( gtk_entry_get_text(GTK_ENTRY(airbrush_text)) );
	xoffset = atoi( gtk_entry_get_text(GTK_ENTRY(offset_x_text)) );
	yoffset = atoi( gtk_entry_get_text(GTK_ENTRY(offset_y_text)) );
	xmargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_x_text)) );
	ymargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_y_text)) );

	/*
	 * Validate data.
	 */
	if (scaling_factor > SCALING_MAX_DEFAULT) 
	{
		scaling_factor = SCALING_MAX_DEFAULT;
		sprintf(buf,"%.1f",scaling_factor);
		gtk_entry_set_text(GTK_ENTRY(scaling_text), buf);
	}
	if (dpi > DPI_MAX_DEFAULT) 
	{
		dpi = DPI_MAX_DEFAULT;
		sprintf(buf,"%d", dpi);
		gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	}
	if (xmargin > MARGIN_MAX_DEFAULT) 
	{
		xmargin = MARGIN_MAX_DEFAULT;
		sprintf(buf,"%d", xmargin);
		gtk_entry_set_text(GTK_ENTRY(margin_x_text), buf);
	}
	if (ymargin > MARGIN_MAX_DEFAULT) 
	{
		ymargin = MARGIN_MAX_DEFAULT;
		sprintf(buf,"%d", ymargin);
		gtk_entry_set_text(GTK_ENTRY(margin_y_text), buf);
	}

	layer_id = gimp_image_get_active_layer(image_id);

	/*
	 * Convert degrees into radians.
	 */
	angle_radians = (angle_degrees + 90) * ( 2 * M_PI / 360 );

	/*
	 * Validate the image id - it may have gone away since the time we
	 * last updated the menu.
	 */
	images = gimp_image_list(&nimages);
	found = FALSE;
	if ( images != NULL )
	{
		for (i=0; i<nimages; i++ )
		{
			if ( images[i] == image_id )
				found = TRUE;
		}
		g_free(images);
	}

	if (found == FALSE)
	{
		sprintf(buf, "%s %s",
			"The selected image seems to have been deleted.  ",
			"Please select another image.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);

		/*
		 * Rebuild the images menu.
		 */
		GFXRebuildImageMenu();

		return;
	}

	/*
	 * Compute the coordinates we need to draw the arrows.
	 */
	GFXComputePts(
		&floatpts, &tailpts, 
		&box_height, &box_width,
		&scaling_factor);

	/*
	 * Group all of these into a single undo.
	 */
	return_vals = gimp_run_procedure ("gimp_undo_push_group_start",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);


	/*
	 * If requested, create a new layer, scaled to fit the size of the
	 * arrow, and make it the active layer.
	 */
	switch(active_type)
	{
		case ARROWGFX_TYPE_NEW_LAYER:
			/*
			 * First, get the base image type.
			 */
			base_type = (int)gimp_image_base_type(image_id);
			switch(base_type)
			{
				case GIMP_RGB:		base_type = GIMP_RGBA_IMAGE; break;
				case GIMP_GRAY:		base_type = GIMP_GRAYA_IMAGE; break;
				case GIMP_INDEXED:	base_type = GIMP_INDEXEDA_IMAGE; break;
			}

			/*
			 * Add margins to box dimensions.
			 */
			box_width = dpi * scaling_factor + 2*xmargin;
			box_height = dpi * scaling_factor + 2*ymargin;

			/*
			 * Then allocate a new layer, scaled to the right size.
			 */
			layer_id = gimp_layer_new(
							image_id, "GFXArrow", box_width, box_height,
							base_type, opacity, blend_mode);

			gimp_image_add_layer(image_id, layer_id, -1);
			gimp_layer_add_alpha(layer_id);

			/*
			 * Initialize layer (all transparent).
			 */
			drawable = gimp_drawable_get (layer_id);
			gimp_pixel_rgn_init (
				&pixel_rgn, 
				drawable, 
				0, 0, 
				drawable->width,
				drawable->height, 
				TRUE, FALSE);
			pixel_block = 
				(unsigned char *)g_malloc (box_width * box_height * 1.5 * 4);
			memset(pixel_block, 0x00, box_width * box_height * 1.5 * 4);
			gimp_pixel_rgn_set_rect (
				&pixel_rgn,
				pixel_block,
				0,
				0,
				box_width,
				box_height);
			g_free(pixel_block); 

			/*
			 * Make it the active layer and turn off Preserve Transparency.
			 */
			gimp_image_set_active_layer(image_id, layer_id);
			gimp_layer_set_preserve_transparency(layer_id, (int)FALSE);

			break;

		case ARROWGFX_TYPE_ANCHOR:
			/* Do nothing here - just draw in the current active drawable */
			break;

		case ARROWGFX_TYPE_MASK:
			/*
			 * Check for existing mask. If one exists, make sure its active.
			 * If not, create a new one and make that one active.
			 */
			mask_id = gimp_layer_get_mask_id(layer_id);
			if (mask_id == -1)
			{
				mask_id = gimp_layer_create_mask(layer_id, 0);
				gimp_image_add_layer_mask(image_id, layer_id, mask_id);
				gimp_image_set_active_channel(image_id, mask_id);
				gimp_image_set_active_layer(image_id, layer_id);
			}
			else
				gimp_image_set_active_channel(image_id, mask_id);
			break;

		case ARROWGFX_TYPE_CHANNEL:
			break;

	}

	/*
	 * Get the active drawable for the selected image.
	 */
	return_vals = gimp_run_procedure ("gimp_image_active_drawable",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	active_drawable = return_vals[1].data.d_drawable;
	gimp_destroy_params (return_vals, nreturn_vals);
	DBGPrintf(DBG_INFO,("drawable id: %d\n", active_drawable));

	/*
	 * Now we're ready to draw.  The procedure used to draw the outline
	 * depends on the tool_type variable.
	 */
	if ( fill_type != ARROWGFX_TYPE_PATTERN )
	{
		switch (tool_type)
		{
			case ARROWGFX_TYPE_PENCIL:
				return_vals = gimp_run_procedure ("gimp_pencil",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, (arrow_hdrs[selected_index-1].points+1)*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
				if ( tail_status == ARROWGFX_TYPE_DRAW )
				{
					return_vals = gimp_run_procedure ("gimp_pencil",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, 5*2,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_END);
					gimp_destroy_params (return_vals, nreturn_vals);
				}
				break;

			case ARROWGFX_TYPE_PAINTBRUSH:
				return_vals = gimp_run_procedure ("gimp_paintbrush",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_FLOAT, fade,
							GIMP_PDB_INT32, (arrow_hdrs[selected_index-1].points+1)*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_INT32, 1,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
				if ( tail_status == ARROWGFX_TYPE_DRAW )
				{
					return_vals = gimp_run_procedure ("gimp_paintbrush",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_FLOAT, fade,
							GIMP_PDB_INT32, 10,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_INT32, 1,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
					gimp_destroy_params (return_vals, nreturn_vals);
				}
				break;

			case ARROWGFX_TYPE_AIRBRUSH:
#ifdef DEBUG
				DBGPrintf(DBG_INFO,("drawable id   : %d\n", active_drawable));
				DBGPrintf(DBG_INFO,("pressure      : %f\n", pressure));
				DBGPrintf(DBG_INFO,("selected_index: %d\n", selected_index));
				DBGPrintf(DBG_INFO,("num_points    : %d\n", 
						(arrow_hdrs[selected_index-1].points+1)*2));
				for (i=0;i<arrow_hdrs[selected_index-1].points+1;i++)
				{
					DBGPrintf(DBG_INFO,("\tvertice: %f x %f\n", 
							floatpts[i*2], floatpts[i*2+1]));
				}
#endif
				return_vals = gimp_run_procedure ("gimp_airbrush",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_FLOAT, pressure,
							GIMP_PDB_INT32, (arrow_hdrs[selected_index-1].points+1)*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);

				if ( tail_status == ARROWGFX_TYPE_DRAW )
				{
					return_vals = gimp_run_procedure ("gimp_airbrush",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_FLOAT, pressure,
							GIMP_PDB_INT32, 10,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_END);
					gimp_destroy_params (return_vals, nreturn_vals);
				}
				break;

			case ARROWGFX_TYPE_STROKE:
				return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, (arrow_hdrs[selected_index-1].points+1)*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_INT32, 2,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
				if ( tail_status == ARROWGFX_TYPE_DRAW )
				{
					return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, 4*2,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_INT32, 0,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
					gimp_destroy_params (return_vals, nreturn_vals);
				}
				return_vals = gimp_run_procedure ("gimp_edit_stroke", 
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
				return_vals = gimp_run_procedure ("gimp_selection_clear", 
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
				break;
		}
	}

	/*
	 * Now check to see if we need to fill the arrow in with anything.
	 */
	switch(fill_type)
	{
		case ARROWGFX_TYPE_FILL:
			return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, arrow_hdrs[selected_index-1].points*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_INT32, 2,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			if ( tail_status == ARROWGFX_TYPE_DRAW )
			{
				return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, 4*2,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_INT32, 0,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
			}
			return_vals = gimp_run_procedure ("gimp_bucket_fill",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 100.0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			return_vals = gimp_run_procedure ("gimp_selection_clear", 
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			break;

		case ARROWGFX_TYPE_PATTERN_OL:
		case ARROWGFX_TYPE_PATTERN:
			return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, arrow_hdrs[selected_index-1].points*2,
							GIMP_PDB_FLOATARRAY, floatpts,
							GIMP_PDB_INT32, 2,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			if ( tail_status == ARROWGFX_TYPE_DRAW )
			{
				return_vals = gimp_run_procedure ("gimp_free_select",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_INT32, 4*2,
							GIMP_PDB_FLOATARRAY, tailpts,
							GIMP_PDB_INT32, 0,	
							GIMP_PDB_INT32, 1,	
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
				gimp_destroy_params (return_vals, nreturn_vals);
			}
			return_vals = gimp_run_procedure ("gimp_bucket_fill",
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, 2,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 100.0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_FLOAT, 0.0,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			return_vals = gimp_run_procedure ("gimp_selection_clear", 
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
			gimp_destroy_params (return_vals, nreturn_vals);
			break;

		case ARROWGFX_TYPE_OUTLINE:
			break;
	}

	/*
	 * Offset the layer, if necessary.
	 */
	if ( active_type == ARROWGFX_TYPE_NEW_LAYER )
	{
		if ( (xoffset > 0) || (yoffset > 0) )
			gimp_layer_set_offsets(layer_id, xoffset, yoffset);
	}

	/*
	 * End the Undo Group.
	 */
	return_vals = gimp_run_procedure ("gimp_undo_push_group_end",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);


	/*
	 * Free up allocated memory.
	 */
	g_free(floatpts);
	g_free(tailpts);
	
	/*
	 * Rebuild the images menu.
	 */
	GFXRebuildImageMenu();

	/*
	 * Update windows.
	 */
	gimp_displays_flush ();

	/* 
	 * Possible Gimp bug:
	 * I have to set the active layer here because of strange behaviour in
	 * setting the mask active, drawing to it and making sure the masks
	 * layer is set to be active.  I can do it all *except* setting the
	 * active layer before the first gimp_displays_flush().  If I try to set the
	 * active layer *before* that the mask does not get updated.  If I don't add
	 * the additional gimp_displays_flush() *after* setting the active layer 
	 * then the layer doesn't get displayed as active (blue highlight in Layers
	 * and Channels dialog).  Weird, but maybe thats fixed in the Gimp 1.2 tree.
	 */
	gimp_image_set_active_layer(image_id, layer_id);
	gimp_displays_flush ();
}

