/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: page1.c
 *            DESCRIPTION: notebook page 1 related routines
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define PAGE1_C

/* === System Headers === */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <math.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "arrows.h"
#include "arrowheads.h"
#include "gfxcommon.h"
#include "debug.h"


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();
extern void GFXComputePts();


/* === Public routine prototypes === */
void GFXPreviewUpdate();
void GFXPreviewButtonPress();
void GFXPreviewButtonRelease();
void GFXPreviewMotion();
void GFXTypeToggle();
void GFXSizeSelect();
void GFXArrowCallback();
void GFXTextEntryUpdates();
void GFXDPITextUpdate();
void GFXDPISliderUpdate();
void GFXScalingTextUpdate();
void GFXScalingSliderUpdate();


/* === Private routine prototypes === */


/* === Global Variables === */
extern int selected_index;
extern int tail_status;

extern GtkDrawingArea	*preview;
extern GtkWidget		*blend_menu;
extern GtkWidget		*opacity_text;
extern GtkWidget		*opacity_slider;
extern GtkWidget		*tail_length_text;
extern GtkObject		*tail_length_adj;
extern GtkWidget		*scaling_text;
extern GtkWidget		*scaling_slider;
extern GtkObject		*scaling_adj;
extern GtkWidget		*margin_y_text;
extern GtkWidget		*margin_x_text;
extern GtkWidget		*dpi_text;
extern GtkWidget		*dpi_slider;
extern GtkObject		*dpi_adj;
extern GtkWidget		*size_menu;
extern GtkWidget		*size_label;
extern GtkWidget		*new_layer_radio_button;
extern GtkWidget		*angle_text;

extern int	mouse_x, mouse_y;		/* current location of mouse on mouse clicks */
extern GdkPoint	endpoint;		/* XY coords for corners of page in preview */
extern int		active_type;		/* new layer or anchor to original? */
extern gfloat	rotation_angle;	/* current rotation amount */

int		size_type = ARROWGFX_INCHES;	/* inches or pixels */
gfloat	keymod = 1.0;			/* modifier for rotation calculations */
int		arrow_rotate;			/* if True: mouse drags rotate page */
int		keypress = FALSE;


/*========================================================================
 *	Name:			GFXTypeToggle
 *	Prototype:	void GFXTypeToggle(GtkWidget *widget, gpointer data)
 *					
 *	Description:
 *		Set the active type based on which toggle has been selected.
 *
 *	Input Arguments:
 * GtkWidget	*widget		unused
 * gpointer		data			one of
 *									ARROWGFX_TYPE_NEW_LAYER
 *									ARROWGFX_TYPE_ANCHOR
 *									ARROWGFX_TYPE_MASK
 *									ARROWGFX_TYPE_CHANNEL
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXTypeToggle(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch ((int)data)
	{
		case ARROWGFX_TYPE_NEW_LAYER: 
			active_type = ARROWGFX_TYPE_NEW_LAYER; 
			gtk_widget_set_sensitive(blend_menu, TRUE);
			gtk_widget_set_sensitive(opacity_slider, TRUE);
			gtk_widget_set_sensitive(opacity_text, TRUE);
			break;

		case ARROWGFX_TYPE_ANCHOR:    
			active_type = ARROWGFX_TYPE_ANCHOR; 
			gtk_widget_set_sensitive(blend_menu, FALSE);
			gtk_widget_set_sensitive(opacity_slider, FALSE);
			gtk_widget_set_sensitive(opacity_text, FALSE);
			break;

		case ARROWGFX_TYPE_MASK:    
			active_type = ARROWGFX_TYPE_MASK; 
			gtk_widget_set_sensitive(blend_menu, FALSE);
			gtk_widget_set_sensitive(opacity_slider, FALSE);
			gtk_widget_set_sensitive(opacity_text, FALSE);
			break;

		case ARROWGFX_TYPE_CHANNEL:    
			active_type = ARROWGFX_TYPE_CHANNEL; 
			gtk_widget_set_sensitive(blend_menu, FALSE);
			gtk_widget_set_sensitive(opacity_slider, FALSE);
			gtk_widget_set_sensitive(opacity_text, FALSE);
			break;
	}

	GFXPreviewUpdate();
}


/*========================================================================
 *	Name:			GFXPreviewButtonPress
 *	Prototype:	void GFXPreviewButtonPress()
 *					
 *	Description:
 *		Determine mouse coordinates when the mouse button is pressed
 *		in the drawing area (preview) widget.  If the mouse press is within
 *		a given distance to the rotate grab box then turn on rotation.
 *		otherwise make sure its turned off.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXPreviewButtonPress(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * Turn on rotations.
	 */
	arrow_rotate = TRUE;

	/*
	 * Update the preview window.
	 */
	GFXPreviewUpdate();
}


/*========================================================================
 *	Name:			GFXPreviewButtonRelease
 *	Prototype:	void GFXPreviewButtonRelease()
 *					
 *	Description:
 *		Disable rotation when user releases button.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXPreviewButtonRelease(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	arrow_rotate = FALSE;
}


/*========================================================================
 *	Name:			GFXPreviewUpdate
 *	Prototype:	void GFXPreviewUpdate()
 *					
 *	Description:
 *		Redraw the preview window.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXPreviewUpdate()
{
	int				xoffset, yoffset;
	static GdkGC	*gcWhite = NULL;
	static GdkGC	*gcBlack = NULL;
	static GdkGC	*gcDotted = NULL;
	gfloat			radians;
	GdkPoint			adj;
	char				buf[32];
	GdkPoint			*pts;
	GdkPoint			tail[4];
	int				i;
	int				height;

	static GdkPixmap		*offscreen_pixmap=NULL;

	if ( preview->widget.window == NULL )
		return;

	/*
	 * Generate an offscreen pixmap.  We draw in this and then copy it to
	 * the real window in order to eliminate flashing - thats called double
	 * buffering.
	 */
	if ( offscreen_pixmap == NULL )
	{
		offscreen_pixmap = 
			gdk_pixmap_new(
				GTK_WIDGET(preview)->window,
				GTK_WIDGET(preview)->allocation.width,
				GTK_WIDGET(preview)->allocation.height,
				-1);
	}

	/*
	 * Clear the offscreen pixmap.
	 */
	gdk_draw_rectangle(
		offscreen_pixmap,
		GTK_WIDGET(preview)->style->bg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		TRUE,
		0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height);

	if (gcWhite == NULL)
	{
		gcWhite = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcWhite, GTK_WIDGET(preview)->style->white_gc );
		gcBlack = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcBlack, GTK_WIDGET(preview)->style->black_gc );
		gcDotted = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcDotted, GTK_WIDGET(preview)->style->black_gc );
	}

	/*
	 * Draw external circle inside which the arrow is rotated.
	 */
	xoffset = 2;
	yoffset = 2;
	gdk_draw_arc(
		offscreen_pixmap, gcBlack, FALSE,
		xoffset,yoffset,
		PREVIEW_WSIZE-4, PREVIEW_HSIZE-4,
		0, 360*64);

	/*
	 * Compute coordinate of the end of the arrow.
	 */
	radians = rotation_angle * ( 2.0 * M_PI / 360.0 );

	endpoint.x = cos(radians) * ((PREVIEW_WSIZE - 4) / 2);
	endpoint.y = sin(radians) * ((PREVIEW_HSIZE - 4) / 2);

	/*
	 * Recompute currently selected arrowheads pts based on rotation.
	 */
	if ( selected_index != 0 )
	{
		radians = (90+rotation_angle) * ( 2.0 * M_PI / 360.0 );

		pts = (GdkPoint *)g_new(gint, 
					arrow_hdrs[selected_index-1].points * sizeof(GdkPoint));

		for(i=0; i<arrow_hdrs[selected_index-1].points; i++)
		{
			/*
			 * Image origin is upper left corner and all have their "tips"
			 * a x + grid_size/2.  So we move the whole thing over by
			 * -grid_size/2 before doing any rotations or other transforms.
			 */
			adj.x = (gint16)arrow_vertices[selected_index-1][i].x - 
							(arrow_hdrs[selected_index-1].grid_size / 2);
			adj.y = (gint16)arrow_vertices[selected_index-1][i].y;

			/*
			 * Now rotate image to point in direction of rotation.
			 */
			pts[i].x = (adj.x * cos(radians)) - (adj.y * sin(radians));
			pts[i].y = (adj.x * sin(radians)) + (adj.y * cos(radians));

			/*
			 * Move image "tip" to end of drawn line.
			 */
			pts[i].x += ((PREVIEW_WSIZE - 4) / 2) + xoffset + endpoint.x;
			pts[i].y += ((PREVIEW_WSIZE - 4) / 2) + yoffset + endpoint.y;
		}

		/*
		 * Draw arrow head.
		 */
		gdk_draw_polygon(
			offscreen_pixmap, gcBlack, TRUE,
			pts, arrow_hdrs[selected_index-1].points);

		g_free(pts);

		/*
		 * Computer tail coordinates.
		 */
		tail[0].x = (gint16)arrow_hdrs[selected_index-1].tail.x1;
		tail[0].y = (gint16)arrow_hdrs[selected_index-1].tail.y1;
		tail[1].x = (gint16)arrow_hdrs[selected_index-1].tail.x2;
		tail[1].y = (gint16)arrow_hdrs[selected_index-1].tail.y2;
	
		height = atoi(gtk_entry_get_text(GTK_ENTRY(tail_length_text)));
	
		/* 
		 * Other two points have to follow clockwise.
		 */
		tail[2].x = tail[1].x;
		tail[2].y = tail[1].y + height;
		tail[3].x = tail[0].x;
		tail[3].y = tail[0].y + height;

		for (i=0; i<4; i++)
		{
			/* Offset to center on original grid */
			adj.x = (gint16)tail[i].x -
							(arrow_hdrs[selected_index-1].grid_size / 2);
			adj.y = (gint16)tail[i].y;

			/*
			 * Rotate tail to point in direction of rotation.
			 */
			tail[i].x = (adj.x * cos(radians)) - (adj.y * sin(radians));
			tail[i].y = (adj.x * sin(radians)) + (adj.y * cos(radians));

			/*
			 * Move image "tip" to proper location by arrowhead.
			 */
			tail[i].x += ((PREVIEW_WSIZE - 4) / 2) + xoffset + endpoint.x;
			tail[i].y += ((PREVIEW_WSIZE - 4) / 2) + yoffset + endpoint.y;
		}

		/* Draw the tail */
		if ( tail_status == ARROWGFX_TYPE_DRAW )
		{
			gdk_draw_polygon( offscreen_pixmap, gcBlack, TRUE, tail, 4);
		}
	}

	/*
	 * Copy pixmap into the preview drawing area.
	 */
	gdk_draw_pixmap(
		GTK_WIDGET(preview)->window,
		GTK_WIDGET(preview)->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		offscreen_pixmap,
		0,0, 0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height
		);


	/*
	 * Update angle displayed in text entry field. We need to block the
	 * callbacks for this field first, update the field, then unblock
	 * the callbacks or else we get stuck in an infinite loop.
	 */
	if ( !keypress )
	{
		gtk_signal_handler_block_by_func (GTK_OBJECT (angle_text), 
			(GtkSignalFunc) GFXTextEntryUpdates, (gpointer)NULL);
		sprintf(buf, "%.3f", (gfloat)rotation_angle);
		gtk_entry_set_text(GTK_ENTRY(angle_text), buf);
		gtk_signal_handler_unblock_by_func (GTK_OBJECT (angle_text), 
			(GtkSignalFunc) GFXTextEntryUpdates, (gpointer)NULL);
	}


	/*
	 * Update size info.
	 */
	if (selected_index != 0 )
		GFXSizeSelect(0, (gpointer)size_type);

	/*
	 * Force screen updates.
	 */
	gdk_flush();

}


/*========================================================================
 *	Name:			GFXPreviewMotion
 *	Prototype:	void GFXPreviewMotion()
 *					
 *	Description:
 *		Update preview window settings based on which item (a margin line
 *		or the grab box) is going to be moved.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXPreviewMotion(
	GtkWidget		*widget,
	GdkEventMotion	*event
)
{
	int		offsetx, offsety;

	/*
	 * Get current mouse location and compute its offset from when the mouse
	 * button was pressed.
	 */
	offsetx = event->x - mouse_x;
	offsety = event->y - mouse_y;

	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * Check for keyboard modifiers.
	 */
	keymod = 1.0;
	if (event->state & GDK_SHIFT_MASK)
	{
		keymod = 3.0;
	}
	if (event->state & GDK_CONTROL_MASK)
	{
		keymod = 6.0;
	}
	if ( (event->state & GDK_CONTROL_MASK) && (event->state & GDK_SHIFT_MASK) )
	{
		keymod = 10.0;
	}

	/*
	 * If a grab box is enabled, compute new angle.
	 */
	if ( arrow_rotate )
	{
		rotation_angle += offsetx/keymod + offsety/keymod;
		if ( rotation_angle >  359.0 ) rotation_angle = 0.0;
		if ( rotation_angle < -359.0 ) rotation_angle = 0.0;
	}

	/*
	 * Redraw the screen.
	 */
	GFXPreviewUpdate();

}


/*========================================================================
 *	Name:			GFXSizeSelect
 *	Prototype:	void GFXSizeSelect()
 *					
 *	Description:
 *	Toggle between inches and pixels for size calculations.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXSizeSelect(
	gint32 	id, 
	gpointer	data
)
{
	gfloat		dpi;
	gfloat		scaling_factor;
	int			xmargin, ymargin;
	int			box_width, box_height;
	char			buf[128];

	size_type = (int) data;
	printf("size type: %d\n", size_type);

	/*
	 * Don't do anything without first making sure an arrowhead has been
	 * selected.
	 */
	if ( selected_index == 0 )
	{
		sprintf(buf, "%s", "You need to select an arrow type first.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);
		return;
	}

	/*
	 * Recalculate displayed size.
	 */
	xmargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_x_text)) );
	ymargin = atoi( gtk_entry_get_text(GTK_ENTRY(margin_y_text)) );
	dpi = atoi( gtk_entry_get_text(GTK_ENTRY(dpi_text)) );

	GFXComputePts(NULL, NULL, &box_width, &box_height, &scaling_factor);

	box_width = dpi * scaling_factor + 2*xmargin;
	box_height = dpi * scaling_factor + 2*ymargin;

	switch (size_type)
	{
		case ARROWGFX_INCHES:
			sprintf(buf, "%.3f x %.3f", (box_width/dpi), (box_height/dpi));
			break;

		case ARROWGFX_PIXELS:
			sprintf(buf, "%d x %d", box_width, box_height);
			break;
	}
	gtk_label_set_text(GTK_LABEL(size_label), buf);
}


/*========================================================================
 *	Name:			GFXArrowCallback
 *	Prototype:	void GFXArrowCallback()
 *					
 *	Description:
 *		Handle mouse clicks on arrow buttons.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXArrowCallback(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch((int)data)
	{
		case ARROWGFX_DOWN: 
			rotation_angle += 0.001;
			if ( rotation_angle > 359.0 ) 
				rotation_angle = 0.0;
			break;

		case ARROWGFX_UP: 
			rotation_angle -= 0.001;
			if ( rotation_angle < -359.0 ) 
				rotation_angle = 0.0;
			break;
	}

	GFXPreviewUpdate();
}


/*========================================================================
 *	Name:			GFXTextEntryUpdates
 *	Prototype:	void GFXTextEntryUpdates()
 *					
 *	Description:
 *		Update the display based on input from the user.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXTextEntryUpdates(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	angle;

	/*
	 * Block signals and prevent other routines from updating the
	 * input field  while we're doing this update.
	 */
	gtk_signal_handler_block_by_func (GTK_OBJECT (angle_text), 
		(GtkSignalFunc) GFXTextEntryUpdates, (gpointer)NULL);
	keypress = TRUE;

	/*
	 * Get Angle text data.
	 */
	angle = atof( gtk_entry_get_text(GTK_ENTRY(angle_text)) );

	/*
	 * Clamp it between 359 and -359.
	 */
	if ( angle >  359.0 ) angle = 0.0;
	if ( angle < -359.0 ) angle = 0.0;

	rotation_angle = angle;

	GFXPreviewUpdate();

	/*
	 * Allow other updates and signals to proceed now.
	 */
	keypress = FALSE;
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (angle_text), 
		(GtkSignalFunc) GFXTextEntryUpdates, (gpointer)NULL);
}



/*========================================================================
 *	Name:			GFXScalingTextUpdate
 *	Prototype:	void GFXScalingTextUpdate()
 *					
 *	Description:
 *		Handle updates from Scaling text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXScalingTextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	/*
	 * Block signals and prevent other routines from updating the
	 * input field  while we're doing this update.
	 */
	gtk_signal_handler_block_by_func (GTK_OBJECT (scaling_text), 
		(GtkSignalFunc) GFXScalingTextUpdate, (gpointer)NULL);
	keypress = TRUE;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(scaling_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(scaling_adj)->upper))
	{
		GTK_ADJUSTMENT(scaling_adj)->value = newvalue;
		gtk_signal_emit_by_name(scaling_adj, "value_changed");
	}

	keypress = FALSE;
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (scaling_text), 
		(GtkSignalFunc) GFXScalingTextUpdate, (gpointer)NULL);
}


/*========================================================================
 *	Name:			GFXScalingSliderUpdate
 *	Prototype:	void GFXScalingSliderUpdate()
 *					
 *	Description:
 *		Handle updates from Scaling scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXScalingSliderUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	if (keypress)
		return;

	sprintf(buf, "%.1f", adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(scaling_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(scaling_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(scaling_text), NULL);

	/*
	 * Update size display.
	 */
	if (selected_index != 0 )
	{
		GFXSizeSelect(0, (gpointer)size_type);
	}
}


/*========================================================================
 *	Name:			GFXDPITextUpdate
 *	Prototype:	void GFXDPITextUpdate()
 *					
 *	Description:
 *		Handle updates from DPI text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXDPITextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(dpi_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(dpi_adj)->upper))
	{
		GTK_ADJUSTMENT(dpi_adj)->value = newvalue;
		gtk_signal_emit_by_name(dpi_adj, "value_changed");
	}
}


/*========================================================================
 *	Name:			GFXDPISliderUpdate
 *	Prototype:	void GFXDPISliderUpdate()
 *					
 *	Description:
 *		Handle updates from DPI scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXDPISliderUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	sprintf(buf, "%d", (int)adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(dpi_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(dpi_text), NULL);

	/*
	 * Update size display.
	 */
	if (selected_index != 0 )
		GFXSizeSelect(0, (gpointer)size_type);
}


/*========================================================================
 *	Name:			GFXTailTextUpdate
 *	Prototype:	void GFXTailTextUpdate()
 *					
 *	Description:
 *		Handle updates from Tail text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXTailTextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(tail_length_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(tail_length_adj)->upper))
	{
		GTK_ADJUSTMENT(tail_length_adj)->value = newvalue;
		gtk_signal_emit_by_name(tail_length_adj, "value_changed");
	}
}


/*========================================================================
 *	Name:			GFXTailSliderUpdate
 *	Prototype:	void GFXTailSliderUpdate()
 *					
 *	Description:
 *		Handle updates from Tail scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXTailSliderUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	sprintf(buf, "%d", (int)adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(tail_length_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(tail_length_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(tail_length_text), NULL);

	/*
	 * Update size display.
	 */
	if (selected_index != 0 )
	{
		GFXSizeSelect(0, (gpointer)size_type);
		GFXPreviewUpdate();
	}
}

