/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: bcards.c
 *            DESCRIPTION: main module for Business Cards plug-in
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define MAIN_C		/* needed by debug library */

/* === System Headers === */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "bcards.h"
#include "gfxcommon.h"
#include "gm-logo.xpm"
#include "debug.h"


/*
 * _GimpPlugInInfo is a structure that tells the gimp what routines to
 * run at certain times.  For most plug-ins, init_proc() and 
 * quit_proc() are unnecessary.  All plug-ins will want to define
 * a query_proc() and a run_proc().
 */
typedef struct _GimpPlugInInfo GPlugInfo;


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();

extern void BCardsImageSetup();
extern void BCardsImageSelect();
extern void BCardsResetFields();
extern void BCardsTextEntryUpdates();
extern void GFXRebuildImageMenu();


/* === Public routine prototypes === */
static void BCardsQuery();	/* static, but made global with PLUG_IN_INFO */
static void BCardsRun();	/* static, but made global with PLUG_IN_INFO */


/* === Private routine prototypes === */
static void BCards();		/* the heart of the plug-in */
static void CloseCallback();
static void OKCallback();
static void MarginToggle();
static void DPIUpdate();
static void DPITextUpdate();
static void PreviewUpdate();
static void PreviewButtonPress();
static void PreviewButtonRelease();
static void PreviewMotion();
static void ResetFields();
static void ArrowCallback();
static void ComputeSuggestedSize();
static void TextEntryUpdates();
static void ShowStatus();
static void ShowHelp();
static void GridSelect();


/* === Global Variables === */
extern gfloat	imargin_top;
extern gfloat	imargin_bottom;
extern gfloat	imargin_left;
extern gfloat	imargin_right;
extern int		border_state;

extern GtkWidget	*sharpen_text;
extern GtkWidget	*border_check_button;


GimpPlugInInfo PLUG_IN_INFO = {	/* Required:  you *must* name this PLUG_IN_INFO */
	NULL,				/* No Init procedure */
	NULL,				/* No Exit procedure */
	BCardsQuery,	/* Query procedure */
	BCardsRun,		/* Run time procedure - where the work really starts */
};


GtkWidget		*top_radio_button;
GtkWidget		*bcards_dialog;
GtkWidget		*size_info;
GtkWidget		*page_width_text, *page_height_text;
GtkWidget		*row_text, *column_text;
GtkWidget		*row_left_button;
GtkWidget		*row_right_button;
GtkWidget		*col_left_button;
GtkWidget		*col_right_button;
GtkWidget		*top_margin_text;
GtkWidget		*bottom_margin_text;
GtkWidget		*left_margin_text;
GtkWidget		*right_margin_text;
GtkWidget		*src_current_size_text;
GtkWidget		*src_suggested_size_text;
GimpDrawable		*drawable;
GtkWidget		*dpi_text;
GtkObject		*dpi_adj;
GtkDrawingArea	*preview;
GtkWidget		*msg_widget;
GtkWidget		*msg_data;
GtkWidget		*status_label;
GtkWidget		*help_widget;

int		image_width, image_height;
int		drawable_width, drawable_height;
gint32	drawable_id;
gint		*grid_sq=NULL;

int		columns, rows;

int		resize_x, resize_y;	/* hot spot of page resize box */
int		mouse_x, mouse_y;		/* current location of mouse on mouse clicks */
int		page_resize_flag;		/* if True: mouse drags resize page */

gfloat	page_width=PAGE_WIDTH_DEFAULT;		/* current width of page */
gfloat	page_height=PAGE_HEIGHT_DEFAULT;		/* current height of page */

int		active_margin;			/* which margin line is currently editable */

gfloat	margin_top = MARGIN_TOP_DEFAULT;			/* Top margin, in inches */
gfloat	margin_bottom = MARGIN_BOTTOM_DEFAULT;	/* Bottom margin, in inches */
gfloat	margin_left = MARGIN_LEFT_DEFAULT;		/* Left margin, in inches */
gfloat	margin_right = MARGIN_RIGHT_DEFAULT;	/* Right margin, in inches */

int		image_id;	/* ID of image to use as source */
int		drag_state;	/* whether dragging should be done or not */



/*========================================================================
 *	Name:			BCardsQuery
 *	Prototype:	static void BCardsQuery()
 *					
 *	Description:
 *		Register this plug-in in the Gimp Procedural Database.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *	Most of the gimp_install_procedure() arguments are defined values that
 * can be found in the bcards.h file.
 *========================================================================*/
static void
BCardsQuery()
{
	static GimpParamDef args[] = {
		{ GIMP_PDB_INT32,		"run_mode",		"Interactive, non-interactive" },
		{ GIMP_PDB_IMAGE,		"image",			"Input image" },
		{ GIMP_PDB_DRAWABLE,	"drawable",		"Input drawable" },
		{ GIMP_PDB_INT32,		"rows",			"Number of rows" },
		{ GIMP_PDB_INT32,		"columns",		"Number of columns" },
		{ GIMP_PDB_INT32,		"pagewidth",	"Width of printed page (in inches)" },
		{ GIMP_PDB_INT32,		"pageheight",	"Height of printed page (in inches)" },
		{ GIMP_PDB_INT32,		"resolution",		
								"DPI printer will be using (300, 360, 720, etc)" },
		{ GIMP_PDB_INT32,		"margintop",	
								"Margin at top of printed page (in inches)" },
		{ GIMP_PDB_INT32,		"marginbottom",
								"Margin at bottom of printed page (in inches)" },
		{ GIMP_PDB_INT32,		"marginleft",
								"Margin on left side of printed page (in inches)" },
		{ GIMP_PDB_INT32,		"marginright",
								"Margin on right side of printed page (in inches)" },
	};
	static int	nargs = sizeof(args) / sizeof(args[0]);

	gimp_install_procedure(
		BCARDS_NAME,
		BCARDS_BLURB,
		BCARDS_HELP,
		BCARDS_AUTHOR,
		BCARDS_COPYRIGHT,
		BCARDS_DATE,
		BCARDS_MENU_PATH,
		BCARDS_IMAGE_TYPES,
		BCARDS_PROC_TYPE,
		nargs,
		0,
		args,
		NULL
	);
}


/*========================================================================
 *	Name:			BCardsRun
 *	Prototype:	static void BCardsRun()
 *					
 *	Description:
 *		Starts the plug-in.  Basically just preps for creatingg the dialog
 *		and does some initialization before calling the heart of the 
 *		plugin, BCards().
 *
 *	Input Arguments:
 *	char		*name				Name of print program.
 *	int		nparams			Number of parameters passed in
 *	GimpParam	*param			Parameter values
 *
 *	Output Arguments:
 *	int		*nreturn_vals	Number of return values
 *	GimpParam	**return_vals	Return values
 *
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
BCardsRun(
	char		*name,			/* Name of print program. */
	int		nparams,			/* Number of parameters passed in */
	GimpParam	*param,			/* Parameter values */
	int		*nreturn_vals,	/* Number of return values */
	GimpParam	**return_vals	/* Return values */
)
{
#ifdef DEBUG
	char			*fname="BCardsRun()";
#endif

	GimpRunModeType	run_mode;
	GimpParam			*values;

	char				dbgfile[256];

	/*
	 * Set up for debugging.
	 */
#ifdef DEBUG
	sprintf(dbgfile,"/tmp/gfxcards");
	DBGFile=dbgfile;
	GMDebugLevel=DBGStringToInt("0xffff");
	DBGOpen( DEBUG_FD );
	DBGEnter();
#endif

	/*
	 * Initialize parameter data...
	 */
	run_mode = param[0].data.d_int32;
	values = g_new(GimpParam, 1);
	values[0].type          = GIMP_PDB_STATUS;
	values[0].data.d_status = GIMP_PDB_SUCCESS;
	*nreturn_vals = 1;
	*return_vals  = values;
 
	/*
	 * Get drawable.  This is what we will be scaling and duplicating
	 * unless the user selects another one.
	 */
	drawable = gimp_drawable_get(param[2].data.d_drawable);
	drawable_id = param[2].data.d_drawable;
	drawable_width  = drawable->width;
	drawable_height = drawable->height;

	image_id = gimp_drawable_image_id(drawable_id);
	DBGPrintf(DBG_INFO,("image id: %d\n", image_id));


	/*
	 * Do run-mode specific setup.
	 */
	switch(run_mode)
	{
		case GIMP_RUN_INTERACTIVE:
			/*
			gimp_get_data(BCARDS_PI_NAME, &init_values);
			*/

			BCards();
			break;

		case GIMP_RUN_NONINTERACTIVE:
			break;

		case GIMP_RUN_WITH_LAST_VALS:
			/*
			gimp_get_data(BCARDS_PI_NAME, &init_values);
			*/
			break;

		default:
			values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
			break;
	}
}



/*========================================================================
 *	Name:			MAIN
 *	Prototype:	MAIN()
 *					
 *	Description:
 *		Gimp macro that defines a plug-in's main routine.  All plug-ins must
 *		include this.
 *========================================================================*/
MAIN();



/*========================================================================
 *	Name:			BCards
 *	Prototype:	static void BCards()
 *					
 *	Description:
 *		Set up the dialog and display it.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *		Get the requested drawable and save a copy of it.
 *		Prompt user for input options (re: the dialog window).
 *		Calculate the size it should be based on the number of cards on a
 *			sheet of paper.
 *		Scale the drawable's copy.
 *		Create a new image of the appropriate size.
 *		for i<numrows
 *			for j<numrows
 *				Duplicate the drawable's copy and position it 
 *		Save options to rc file
 *
 *	Restrictions:
 *	Notes:
 *	1. You're best bet is to start with an image that doesn't need to be
 *		scaled much.
 *	2. The "resolution" parameter assumes a square resolution 
 *		(eg 300x300 or 720x720).
 *========================================================================*/
static void
BCards()
{
	gint			argc;
	gchar			**argv;
	char			buf[256];
	int			i;

	GtkWidget	*notebook, *toptable;
	GtkWidget	*table, *margins_table, *dimensions_table, *rc_table;
	GtkWidget	*src_table;
	GtkWidget	*vbox, *hbox, *frame, *hsep;
	GtkWidget	*label;
	GtkWidget	*label_pixmap;
	GtkWidget	*arrow;
	GtkWidget	*toggle;
	GtkWidget	*dpi_scale;
	GtkWidget	*reset_button;
	GtkWidget	*ok_button, *cancel_button, *help_button, *status_button;
	GdkPixmap	*logo;
	GdkBitmap	*logo_mask;
	GtkStyle		*style;
	GtkWidget	*select_all, *select_none;

	GSList			*group = NULL;

	/*
	 * Generic initialization:
	 * 0. Fake the argc/argv stuff.
	 * 1. Initialize the GTK toolkit.
	 * 2. parse the Gimp's gtkrc for plug-in specific settings.
	 * 3. Set up to use XSharedMemory, if possible.
	 */
	argc    = 1;
	argv    = g_new(gchar *, 1);
	argv[0] = g_strdup("bcards");
	gtk_init(&argc, &argv);
	gtk_rc_parse(gimp_gtkrc());
	gdk_set_use_xshm(gimp_use_xshm());

	/*
	 * Try to behave in a civil manner should the unexpected occur.
	 */
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);

	/*
	 * Create a new dialog window.
	 */
	bcards_dialog = gtk_dialog_new();
	sprintf(buf, "%s - %s", BCARDS_TITLE, BCARDS_VERSION);
	gtk_window_set_title(GTK_WINDOW(bcards_dialog), buf);
	gtk_window_set_wmclass(GTK_WINDOW(bcards_dialog), "bcards", "Gimp");
	gtk_window_position(GTK_WINDOW(bcards_dialog), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(bcards_dialog), 0);
	gtk_signal_connect(GTK_OBJECT(bcards_dialog), "destroy",
		(GtkSignalFunc)CloseCallback, NULL);

	gtk_widget_realize(bcards_dialog);

	/*
	 * Initialize the pixmap.
	 */
	style = gtk_widget_get_style( bcards_dialog );
	logo = gdk_pixmap_create_from_xpm_d(
					bcards_dialog->window, 
					&logo_mask, 
					&style->bg[GTK_STATE_NORMAL], 
					gm_logo_xpm);

	/*
	 * The main table in which everything else will go.
	 */
	toptable = gtk_table_new(3, 1, FALSE);
	gtk_container_border_width(GTK_CONTAINER(toptable), 1);
	gtk_table_set_col_spacings(GTK_TABLE(toptable), 1);
	gtk_table_set_row_spacings(GTK_TABLE(toptable), 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (bcards_dialog)->action_area),
		toptable, TRUE, TRUE, 0);
	gtk_widget_show(toptable);

	/*
	 * Main Notebook with 2 tabs.
	 */
	notebook = gtk_notebook_new();
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)notebook,
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_widget_show(notebook);


	/*
	 * The table in which the Business Cards components will go.
	 */
	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 8);
	gtk_widget_show(table);

	/*
	 * Add the table to page 1 of the notebook.
	 */
	label = gtk_label_new("Page Options");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, label);

	/*
	 * A small preview window to show the columns/rows/margins setup.
	 */
	frame = gtk_frame_new("Page Adjustments");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);
	preview = (GtkDrawingArea *)gtk_drawing_area_new();
	gtk_drawing_area_size(preview, PREVIEW_WSIZE, PREVIEW_HSIZE);
	gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *)preview);
	gtk_widget_show((GtkWidget *)preview);

	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "expose_event",
			(GtkSignalFunc)PreviewUpdate,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_press_event",
			(GtkSignalFunc)PreviewButtonPress,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_release_event",
			(GtkSignalFunc)PreviewButtonRelease,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "motion_notify_event",
			(GtkSignalFunc)PreviewMotion,
			NULL);

	gtk_widget_set_events((GtkWidget *)preview,
		GDK_EXPOSURE_MASK | GDK_BUTTON_MOTION_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
 

	/*
	 * A smaller table in which we pack page dimension information, except
	 * for the margins.
	 */
	dimensions_table = gtk_table_new(4, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(dimensions_table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(dimensions_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(dimensions_table), 8);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)dimensions_table, 
		1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(dimensions_table);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)vbox, 
		0, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(vbox);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
	gtk_widget_show(frame);
	label_pixmap = gtk_pixmap_new(logo, logo_mask);
	gtk_container_add (GTK_CONTAINER (frame), label_pixmap);
	gtk_widget_show(label_pixmap);

	label = gtk_label_new("Page Dimensions (in inches)");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	/*
	 * Page width/height
	 */
	label = gtk_label_new("Page Width");
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)label, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(label);

	page_width_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.1f", page_width);
	gtk_entry_set_text(GTK_ENTRY(page_width_text), buf);
	gtk_signal_connect (GTK_OBJECT (page_width_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, (gpointer)BCARDS_PAGE_WIDTH_FIELD);
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)page_width_text, 
		1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(page_width_text);

	label = gtk_label_new("Page Height");
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)label, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(label);

	page_height_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.1f", page_height);
	gtk_entry_set_text(GTK_ENTRY(page_height_text), buf);
	gtk_signal_connect (GTK_OBJECT (page_height_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, (gpointer)BCARDS_PAGE_HEIGHT_FIELD);
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)page_height_text, 
		1, 2, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(page_height_text);

	/*
	 * Rows and Columns
	 */
	rc_table = gtk_table_new(3, 4, FALSE);
	gtk_container_border_width(GTK_CONTAINER(rc_table), 0);
	gtk_table_set_col_spacings(GTK_TABLE(rc_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(rc_table), 8);
	gtk_table_attach(GTK_TABLE(dimensions_table), (GtkWidget *)rc_table, 
		0, 2, 3, 4, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(rc_table);

	label = gtk_label_new("Rows");
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)label, 
		0, 1, 0, 1, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(label);

	row_left_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (row_left_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (row_left_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)1);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)row_left_button, 
		1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(row_left_button);

	arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (row_left_button), arrow);
	gtk_widget_show(arrow);

	sprintf(buf, "%d", ROWS_DEFAULT);
	row_text = gtk_label_new(buf);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)row_text, 
		2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(row_text);
	rows = ROWS_DEFAULT;

	row_right_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (row_right_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (row_right_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)2);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)row_right_button, 
		3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(row_right_button);

	arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (row_right_button), arrow);
	gtk_widget_show(arrow);

	label = gtk_label_new("Columns");
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)label, 
		0, 1, 1, 2, GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_widget_show(label);

	col_left_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (col_left_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (col_left_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)3);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)col_left_button, 
		1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(col_left_button);

	arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (col_left_button), arrow);
	gtk_widget_show(arrow);

	sprintf(buf, "%d", COLUMNS_DEFAULT);
	column_text = gtk_label_new(buf);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)column_text, 
		2, 3, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(column_text);
	columns = COLUMNS_DEFAULT;

	col_right_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (col_right_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (col_right_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)4);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)col_right_button, 
		3, 4, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(col_right_button);

	arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (col_right_button), arrow);
	gtk_widget_show(arrow);

	/*
	 * Allocate the initial array of grid squares, used to determine if
	 * a given square will have a copy of the source image copied to it.
	 */
	grid_sq = (int *)g_malloc(4*columns*rows);
	for (i=0; i<(rows*columns); i++)
		grid_sq[i] = (int)OFF;

	/*
	 * Buttons for selecting all and clearing all grid selections.
	 */
	frame = gtk_frame_new("Grid Square Selections");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(rc_table), (GtkWidget *)frame, 
		0, 4, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(TRUE, 8);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	select_all = gtk_button_new_with_label("Select All");
	gtk_box_pack_start (GTK_BOX (hbox), select_all, TRUE, TRUE, 0);
	gtk_widget_set_usize( select_all, 90, -1 );
	gtk_signal_connect (GTK_OBJECT (select_all), "clicked",
		(GtkSignalFunc) GridSelect, (gpointer)ON);
	gtk_widget_show(select_all);

	select_none = gtk_button_new_with_label("Clear All");
	gtk_box_pack_start (GTK_BOX (hbox), select_none, TRUE, TRUE, 0);
	gtk_widget_set_usize( select_none, 90, -1 );
	gtk_signal_connect (GTK_OBJECT (select_none), "clicked",
		(GtkSignalFunc) GridSelect, (gpointer)OFF);
	gtk_widget_show(select_none);


	/*
	 * Image Windows menu - used to select which image to use when creating
	 * the full page of cards.
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)hbox, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new("Source Image");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	option_menu = gtk_option_menu_new();
	gtk_box_pack_start (GTK_BOX (hbox), option_menu, TRUE, TRUE, 0);
	gtk_widget_show(option_menu);

	menu = gimp_image_menu_new(ImageConstraints,
					BCardsImageSelect,
					NULL,
					image_id);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);
	 */


	/*
	 * DPI input
	 */
	frame = gtk_frame_new("Printers DPI");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	dpi_text = gtk_entry_new_with_max_length(4);
	gtk_box_pack_start (GTK_BOX (hbox), dpi_text, TRUE, TRUE, 0);
	sprintf(buf, "%d", PRINTER_DPI_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	gtk_signal_connect(GTK_OBJECT(dpi_text), "changed",
		(GtkSignalFunc)DPITextUpdate, NULL);
	gtk_widget_show(dpi_text);

	dpi_adj = 
		gtk_adjustment_new(PRINTER_DPI_DEFAULT, 50.0, 1500.0, 1.0, 10.0, 10.0);
	dpi_scale = gtk_hscale_new(GTK_ADJUSTMENT(dpi_adj));
	gtk_box_pack_start (GTK_BOX (hbox), dpi_scale, TRUE, TRUE, 0);
	gtk_widget_set_usize( dpi_scale, 155, -1 );
	gtk_widget_show(dpi_scale);

	gtk_signal_connect(GTK_OBJECT(dpi_adj), "value_changed",
		(GtkSignalFunc)DPIUpdate, NULL);
 

	/*
	 * Suggested source image information.
	 */
	frame = gtk_frame_new("Source Image");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	src_table = gtk_table_new(2, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(src_table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(src_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(src_table), 8);
	gtk_container_add (GTK_CONTAINER (frame), src_table);
	gtk_widget_set_usize( src_table, 205, -1 );
	gtk_widget_show(src_table);

	label = gtk_label_new("Current size");
	gtk_table_attach(GTK_TABLE(src_table), (GtkWidget *)label, 
		0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	src_current_size_text = gtk_label_new("");
	gtk_table_attach(GTK_TABLE(src_table), (GtkWidget *)src_current_size_text, 
		1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(src_current_size_text);
	BCardsImageSelect(image_id, NULL);

	label = gtk_label_new("Suggested size");
	gtk_table_attach(GTK_TABLE(src_table), (GtkWidget *)label, 
		0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	src_suggested_size_text = gtk_label_new("");
	gtk_table_attach(GTK_TABLE(src_table), (GtkWidget *)src_suggested_size_text, 
		1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(src_suggested_size_text);


	/*
	 * Image Size info.
	 */
	frame = gtk_frame_new("Projected New Image Size");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_container_border_width(GTK_CONTAINER(frame), 6);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	size_info = gtk_label_new("");
	gtk_widget_set_usize( size_info, 50, -1 );
	gtk_container_add (GTK_CONTAINER (frame), size_info);
	gtk_widget_show(size_info);


	/*
	 * Margins information go in another table, one that sits
	 * in the middle row of the main table.
	 */
	frame = gtk_frame_new("Page Margins (in inches)");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);

	margins_table = gtk_table_new(2, 6, FALSE);
	gtk_container_border_width(GTK_CONTAINER(margins_table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(margins_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(margins_table), 8);
	gtk_container_add (GTK_CONTAINER (frame), margins_table);
	gtk_widget_show(margins_table);

	label = gtk_label_new("Top");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	top_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", margin_top);
	gtk_entry_set_text(GTK_ENTRY(top_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (top_margin_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, (gpointer)BCARDS_PAGE_TOP_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)top_margin_text, 
		2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(top_margin_text);

	label = gtk_label_new("Bottom");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	bottom_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", margin_bottom);
	gtk_entry_set_text(GTK_ENTRY(bottom_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (bottom_margin_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, 
		(gpointer)BCARDS_PAGE_BOTTOM_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)bottom_margin_text, 
		2, 3, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(bottom_margin_text);

	label = gtk_label_new("Left");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		4, 5, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	left_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", margin_left);
	gtk_entry_set_text(GTK_ENTRY(left_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (left_margin_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, 
		(gpointer)BCARDS_PAGE_LEFT_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)left_margin_text, 
		5, 6, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(left_margin_text);

	label = gtk_label_new("Right");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		4, 5, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	right_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", margin_right);
	gtk_entry_set_text(GTK_ENTRY(right_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (right_margin_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, 
		(gpointer)BCARDS_PAGE_RIGHT_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)right_margin_text, 
		5, 6, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(right_margin_text);


	/*
	 * Radio buttons for each of the margins - these determine which lines
	 * move in the preview.
	 */
	top_radio_button = toggle = gtk_radio_button_new(NULL);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	active_margin = BCARDS_TOPMARGIN;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_TOPMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_BOTTOMMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_LEFTMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		3, 4, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_RIGHTMARGIN);
	gtk_widget_show (toggle);

	/*
	 * Adjust width of all text input fields
	 */
	gtk_widget_set_usize( column_text, 50, -1 );
	gtk_widget_set_usize( row_text, 50, -1 );
	gtk_widget_set_usize( page_height_text, 50, -1 );
	gtk_widget_set_usize( page_width_text, 50, -1 );
	gtk_widget_set_usize( dpi_text, 50, -1 );
	gtk_widget_set_usize( right_margin_text, 50, -1 );
	gtk_widget_set_usize( left_margin_text, 50, -1 );
	gtk_widget_set_usize( bottom_margin_text, 50, -1 );
	gtk_widget_set_usize( top_margin_text, 50, -1 );
/*
	gtk_widget_set_usize( src_current_size_text, 62, -1 );
	gtk_widget_set_usize( src_suggested_size_text, 62, -1 );
*/

	/*
	 * Horizontal separator between main window area and status buttons.
	 */
	hsep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)hsep, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(hsep);

	/*
	 * Add page for image configuration components.
	 */
	BCardsImageSetup(notebook);

	/*
	 * Buttons go along the bottom row of the table.
	 */
	hbox = gtk_hbox_new(TRUE, 4);
	gtk_table_attach(GTK_TABLE(toptable), (GtkWidget *)hbox, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(hbox);

	/*
	 * Ok Button
	 */
	ok_button = gtk_button_new_with_label ("OK");
	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
		(GtkSignalFunc) OKCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), ok_button, TRUE, TRUE, 4);
	gtk_widget_grab_default (ok_button);
	gtk_widget_show(ok_button);

	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (ok_button);

	/*
	 * Cancel Button
	 */
	cancel_button = gtk_button_new_with_label ("Cancel");
	GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		(GtkSignalFunc) CloseCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), cancel_button, TRUE, TRUE, 4);
	gtk_widget_show(cancel_button);

	/*
	 * Reset Button
	 */
	reset_button = gtk_button_new_with_label("Reset");
	GTK_WIDGET_SET_FLAGS (reset_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(reset_button), "clicked",
		(GtkSignalFunc)ResetFields, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), reset_button, TRUE, TRUE, 4);
	gtk_widget_show(reset_button);

	/*
	 * Status Button
	 */
	status_button = gtk_button_new();

	status_label = gtk_label_new("Show Status");
	gtk_container_add (GTK_CONTAINER (status_button), (GtkWidget *)status_label);
	gtk_widget_show(status_label);

	GTK_WIDGET_SET_FLAGS (status_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(status_button), "clicked",
		(GtkSignalFunc)ShowStatus, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), status_button, TRUE, TRUE, 4);
	gtk_widget_show(status_button);

	/*
	 * Help Button
	 */
	help_button = gtk_button_new_with_label ("Help");
	GTK_WIDGET_SET_FLAGS (help_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (hbox), help_button, TRUE, TRUE, 4);
	gtk_signal_connect(GTK_OBJECT(help_button), "clicked",
		(GtkSignalFunc)ShowHelp, NULL);
	gtk_widget_show(help_button);

	/*
	 * Show the main dialog.
	 */
	gtk_widget_show(bcards_dialog);


	/*
	 * Sit and spin.
	 */
	gtk_main();

}


/*========================================================================
 *	Name:			ShowHelp
 *	Prototype:	static void ShowHelp()
 *					
 *	Description:
 *		Manage the Help dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowHelp(
)
{
	if ( help_widget == NULL )
	{
		GFXMsgWindow(
			GFX_HELP_TEXT,
			GFX_INFO_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&help_widget, NULL, 
			NULL, ShowHelp, NULL, ShowHelp, NULL,
			600, 400);

	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (help_widget) )
		{
			gtk_widget_hide(help_widget);
		}
		else
		{
			gtk_widget_show(help_widget);
			GFXCenterWindow(help_widget);
		}
	}
}



/*========================================================================
 *	Name:			ShowStatus
 *	Prototype:	static void ShowStatus()
 *					
 *	Description:
 *		Manage the status dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowStatus(
)
{
	if ( msg_widget == NULL )
	{
		GFXMsgWindow(
			"CardGFX Status\n---------------------\n",
			GFX_MSG_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&msg_widget, &msg_data, 
			NULL, ShowStatus, NULL, ShowStatus, NULL, 0, 0);

		gtk_label_set(GTK_LABEL(status_label), "Hide Status");
	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (msg_widget) )
		{
			gtk_label_set(GTK_LABEL(status_label), "Show Status");
			gtk_widget_hide(msg_widget);
		}
		else
		{
			gtk_label_set(GTK_LABEL(status_label), "Hide Status");
			gtk_widget_show(msg_widget);
			GFXCenterWindow(msg_widget);
		}
	}
}



/*========================================================================
 *	Name:			CloseCallback
 *	Prototype:	static void CloseCallback()
 *					
 *	Description:
 *		Close the main dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
CloseCallback(
)
{
	gimp_drawable_detach(drawable);
	gtk_main_quit();
}


/*========================================================================
 *	Name:			MarginToggle
 *	Prototype:	static void MarginToggle(GtkWidget *widget, gpointer data)
 *					
 *	Description:
 *		Set the active margin line based on which toggle has been selected.
 *
 *	Input Arguments:
 * GtkWidget	*widget		unused
 * gpointer		data			one of
 *									BCARDS_TOPMARGIN
 *									BCARDS_BOTTOMMARGIN
 *									BCARDS_LEFTMARGIN
 *									BCARDS_RIGHTMARGIN
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
MarginToggle(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch ((int)data)
	{
		case BCARDS_TOPMARGIN:    active_margin = BCARDS_TOPMARGIN; break;
		case BCARDS_BOTTOMMARGIN: active_margin = BCARDS_BOTTOMMARGIN; break;
		case BCARDS_LEFTMARGIN:   active_margin = BCARDS_LEFTMARGIN; break;
		case BCARDS_RIGHTMARGIN:  active_margin = BCARDS_RIGHTMARGIN; break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			DPIUpdate
 *	Prototype:	static void DPIUpdate()
 *					
 *	Description:
 *		Handle updates from DPI scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
DPIUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	sprintf(buf, "%.0f", adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(dpi_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(dpi_text), NULL);

	ComputeSuggestedSize();
}


/*========================================================================
 *	Name:			DPITextUpdate
 *	Prototype:	static void DPITextUpdate()
 *					
 *	Description:
 *		Handle updates from DPI text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
DPITextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(dpi_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(dpi_adj)->upper))
	{
		GTK_ADJUSTMENT(dpi_adj)->value = newvalue;
		gtk_signal_emit_by_name(dpi_adj, "value_changed");
	}

	ComputeSuggestedSize();
}


/*========================================================================
 *	Name:			PreviewButtonPress
 *	Prototype:	static void PreviewButtonPress()
 *					
 *	Description:
 *		Determine mouse coordinates when the mouse button is pressed
 *		in the drawing area (preview) widget.  If the mouse press is within
 *		a given distance to the page resize grab box then turn on page
 *		resizing, otherwise make sure its turned off.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonPress(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	gfloat	grid_width, grid_height;
	int		x, y;
	int		xcoord, ycoord;
	int		xoffset, yoffset;

	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * If first button was pressed, we're going to be handling mouse drags
	 * to change page size or margin adjustments.
	 */
	if ( event->button == 1 )
	{
		drag_state = ON;

		/*
		 * Check if user has clicked near the grab box for page resizing.
		 * If so, turn on page resizing.  If not, turn it off.
		 */
		if ( ( abs(mouse_x-resize_x) < 4 ) && ( abs(mouse_y-resize_y) < 4 ) )
			page_resize_flag = TRUE;
		else
			page_resize_flag = FALSE;
	}

	/*
	 * If second or third button was pressed, user is toggling a box to be 
	 * used in the output page.
	 */
	if ( ( event->button == 2 ) || ( event->button == 3 ) )
	{
		drag_state = OFF;

		/*
		 * Calculate which grid square we're in and activate it.
		 */
		grid_width  = 
			(page_width * 10.0) - ( (margin_left + margin_right) * 10.0 );
		grid_height = 
			(page_height * 10.0) - ( (margin_top + margin_bottom) * 10.0 );
		grid_width  /= columns;
		grid_height /= rows;

		xoffset = (PREVIEW_WSIZE-1 - (page_width*10)) / 2;
		yoffset = (PREVIEW_HSIZE-1 - (page_height*10)) / 2;

		for (y=0; y<rows; y++)
		{
			for (x=0; x<columns; x++)
			{
				/* Get coordinates of upper left corner of current grid square. */
				xcoord = (x * grid_width) + (margin_left*10) + xoffset;
				ycoord = (y * grid_height) + (margin_top*10) + yoffset;
				if ( ((mouse_x > xcoord ) && (mouse_x < xcoord+grid_width )) &&
					  ((mouse_y > ycoord ) && (mouse_y < ycoord+grid_height )) )
				{
					if ( grid_sq[(y*columns)+x] == (int)ON )
						grid_sq[(y*columns)+x] = (int)OFF;
					else
						grid_sq[(y*columns)+x] = (int)ON;
				}
			}
		}
	}

	/*
	 * Update the preview window.
	 */
	PreviewUpdate();
}


/*========================================================================
 *	Name:			PreviewButtonRelease
 *	Prototype:	static void PreviewButtonRelease()
 *					
 *	Description:
 *		Reset options when the mouse button is released.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonRelease(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	drag_state = OFF;
}


/*========================================================================
 *	Name:			PreviewUpdate
 *	Prototype:	static void PreviewUpdate()
 *					
 *	Description:
 *		Redraw the preview window, including margin lines and page resize
 *		grab box.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewUpdate()
{
	int				draw_width, draw_height;
	int				xoffset, yoffset;
	int				box_xoffset, box_yoffset;
	int				top_offset, bottom_offset;
	int				left_offset, right_offset;
	static GdkGC	*gcWhite = NULL;
	static GdkGC	*gcBlack = NULL;
	static GdkGC	*gcDotted = NULL;
	static GdkGC	*gcRed = NULL;
	GdkColor			color;
	int				dpi, size;
	char				buf[32];
	int				i;
	int				row_width, column_width;
	int				row_offset, column_offset;
	gfloat			grid_width, grid_height;
	int				x,y;

	static GdkPixmap	*offscreen_pixmap=NULL;


	if ( preview->widget.window == NULL )
		return;

	/*
	 * Generate an offscreen pixmap.  We draw in this and then copy it to
	 * the real window in order to eliminate flashing - thats called double
	 * buffering.
	 */
	if ( offscreen_pixmap == NULL )
	{
		offscreen_pixmap = 
			gdk_pixmap_new(
				GTK_WIDGET(preview)->window,
				GTK_WIDGET(preview)->allocation.width,
				GTK_WIDGET(preview)->allocation.height,
				-1);
	}

	/*
	 * Clear the offscreen pixmap.
	 */
	gdk_draw_rectangle(
		offscreen_pixmap,
		GTK_WIDGET(preview)->style->bg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		TRUE,
		0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height);


	if (gcWhite == NULL)
	{
		/*
		 * This is probably wasteful, but I'm still not clear on color
		 * handling issues with Gtk yet.
		 */
		gcWhite = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcWhite, GTK_WIDGET(preview)->style->white_gc );
		gcBlack = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcBlack, GTK_WIDGET(preview)->style->black_gc );
		gcDotted = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcDotted, GTK_WIDGET(preview)->style->black_gc );

		color.red = 65535;
		color.green = 0;
		color.blue = 0;
		gdk_color_alloc (gtk_widget_get_colormap (GTK_WIDGET(preview)), &color);

		gcRed = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcRed, GTK_WIDGET(preview)->style->fg_gc[0]);
		gdk_gc_set_foreground (gcRed, &color);
		
	}

	/*
	 * Compute the visible page size with respect to the drawing areas physical
	 * region.  Should there be a minimum width and height?  Probably, but
	 * I don't know what to set them to.
	 */
	draw_width = 10 * page_width;
	if ( draw_width > PREVIEW_WSIZE-1 )
		draw_width = PREVIEW_WSIZE-1;
	if ( draw_width < 1 )
		draw_width = 1;

	draw_height = 10 * page_height;
	if ( draw_height > PREVIEW_HSIZE-1 )
		draw_height = PREVIEW_HSIZE-1;
	if ( draw_height < 1 )
		draw_height = 1;

	xoffset = (PREVIEW_WSIZE-1 - draw_width) / 2;
	yoffset = (PREVIEW_HSIZE-1 - draw_height) / 2;

	/*
	 * Draw page.
	 */
	gdk_draw_rectangle( offscreen_pixmap, gcWhite, 1,
		xoffset, yoffset, draw_width, draw_height);

	/*
	 * Update hot spot location for grab box.
	 */
	resize_x = xoffset + draw_width;
	resize_y = yoffset;

	/*
	 * Draw page resize grab box.
	 */
	box_xoffset = xoffset + draw_width - 4;
	box_yoffset = yoffset - 4;

	gdk_draw_rectangle( offscreen_pixmap, gcBlack, 0,
		box_xoffset, box_yoffset, 8, 8);

	/*
	 * Draw margins.  One of these will be active from the toggles, so it
	 * gets drawn in red.  All others are drawn in black.
	 */
	top_offset = yoffset + (margin_top*10);
	bottom_offset = yoffset + draw_height - (margin_bottom*10);
	left_offset = xoffset + (margin_left*10);
	right_offset = xoffset + draw_width - (margin_right*10);

	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Top margin line */
	if ( active_margin == BCARDS_TOPMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line( offscreen_pixmap, gcDotted, 
		0, top_offset, PREVIEW_WSIZE, top_offset );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Bottom margin line */
	if ( active_margin == BCARDS_BOTTOMMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line( offscreen_pixmap, gcDotted, 
		0, bottom_offset, PREVIEW_WSIZE, bottom_offset );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Left margin line */
	if ( active_margin == BCARDS_LEFTMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		left_offset, 0, left_offset, PREVIEW_HSIZE );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Right margin line */
	if ( active_margin == BCARDS_RIGHTMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		right_offset, 0, right_offset, PREVIEW_HSIZE );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/*
	 * Compute locations of row/column lines and draw them.
	 */
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	row_width = ((page_height - margin_top - margin_bottom)* 10.0) / rows;
	column_width = ((page_width - margin_left - margin_right)* 10.0) / columns;
	for (i=1; i<rows; i++)
	{
		row_offset = top_offset + (i * row_width);
		gdk_draw_line(offscreen_pixmap, gcDotted, 
			left_offset, row_offset, right_offset, row_offset );
	}
	for (i=1; i<columns; i++)
	{
		column_offset = left_offset + (i * column_width);
		gdk_draw_line(offscreen_pixmap, gcDotted, 
			column_offset, top_offset, column_offset, bottom_offset );
	}
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/*
	 * Show which grid squares are active.
	 */
	grid_width  = (page_width * 10) - ( (margin_left + margin_right) * 10 );
	grid_height = (page_height * 10) - ( (margin_top + margin_bottom) * 10 );
	grid_width  /= columns;
	grid_height /= rows;
	for (x=0; x<columns; x++)
	{
		for (y=0; y<rows; y++)
		{
			if ( grid_sq[(y*columns)+x] == ON )
			{
				gdk_draw_arc(offscreen_pixmap, gcRed, FALSE,
					left_offset+ ((x+.5) * grid_width), 
					top_offset + ((y+.5) * grid_height),  
					3, 3,  0, 360*64);
			}
		}
	}

	/*
	 * Copy pixmap into the preview drawing area.
	 */
	gdk_draw_pixmap(
		GTK_WIDGET(preview)->window,
		GTK_WIDGET(preview)->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		offscreen_pixmap,
		0,0, 0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height
		);


	/*
	 * Force screen updates.
	 */
	gdk_flush();

	/*
	 * Update the image size window.
	 */
	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));
	size = (page_height * dpi) * (page_width * dpi) / 1024;
	sprintf(buf, "%dK", size);
	gtk_label_set(GTK_LABEL(size_info), buf);

	ComputeSuggestedSize();
}


/*========================================================================
 *	Name:			PreviewMotion
 *	Prototype:	static void PreviewMotion()
 *					
 *	Description:
 *		Update preview window settings based on which item (a margin line
 *		or the grab box) is going to be moved.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewMotion(
	GtkWidget		*widget,
	GdkEventMotion	*event
)
{
	int		offsetx, offsety;
	gfloat	new_page_width, new_page_height;
	gfloat	new_margin;
	char		buf[32];

	if ( drag_state == OFF )
		return;

	/*
	 * Get current mouse location and compute its offset from when the mouse
	 * button was pressed.
	 */
	offsetx = event->x - mouse_x;
	offsety = event->y - mouse_y;

	mouse_x = event->x;
	mouse_y = event->y;


	/*
	 * If the grab box is enabled, update the page size, otherwise update
	 * the margin location.
	 */
	if ( page_resize_flag )
	{
		/*
		 * Calculate new page size.
		 */
		new_page_width = (page_width * 10.0) + offsetx;
		new_page_height = (page_height * 10.0) - offsety;

		/*
		 * Make sure we're not too small or large and that we don't allow
		 * the margins to overlap.
		 */
		if ( new_page_width > PREVIEW_WSIZE-1 )
			page_width = (PREVIEW_WSIZE-1)/10.0;
		else
			page_width = new_page_width/10.0;

		if ( new_page_height > PREVIEW_HSIZE-1 )
			page_height = (PREVIEW_HSIZE-1)/10.0;
		else
			page_height = new_page_height/10.0;

		if ( page_width < (margin_left+margin_right) )
			page_width = margin_left+margin_right;
		if ( page_height < (margin_top+margin_bottom) )
			page_height = margin_top+margin_bottom;


		/*
	 	 * Update page size text fields.
		 */
		sprintf(buf, "%.1f", page_width);
		gtk_entry_set_text(GTK_ENTRY(page_width_text), buf);
		sprintf(buf, "%.1f", page_height);
		gtk_entry_set_text(GTK_ENTRY(page_height_text), buf);

	}
	else
	{
		/*
		 * Calculate new margin position.
		 */
		switch (active_margin)
		{
			case BCARDS_TOPMARGIN:
				new_margin = margin_top * 10.0 + (0.5*offsety);
				margin_top = new_margin/10.0;

				if ( margin_top < 0 )
					margin_top = 0;
				if ( margin_top > (page_height/2) )
					margin_top = page_height/2 - 0.1;
				sprintf(buf, "%.3f", margin_top);
				gtk_entry_set_text(GTK_ENTRY(top_margin_text), buf);
				break;

			case BCARDS_BOTTOMMARGIN:
				new_margin = margin_bottom * 10.0 - (0.5*offsety);
				margin_bottom = new_margin/10.0;

				if ( margin_bottom < 0 )
					margin_bottom = 0;
				if ( margin_bottom > (page_height/2) )
					margin_bottom = page_height/2 - 0.1;
				sprintf(buf, "%.1f", margin_bottom);
				gtk_entry_set_text(GTK_ENTRY(bottom_margin_text), buf);
				break;

			case BCARDS_LEFTMARGIN:
				new_margin = margin_left * 10.0 + (0.5*offsetx);
				margin_left = new_margin/10.0;

				if ( margin_left < 0 )
					margin_left = 0;
				if ( margin_left > (page_width/2) )
					margin_left = page_width/2 - 0.1;
				sprintf(buf, "%.1f", margin_left);
				gtk_entry_set_text(GTK_ENTRY(left_margin_text), buf);
				break;

			case BCARDS_RIGHTMARGIN:
				new_margin = margin_right * 10.0 - (0.5*offsetx);
				margin_right = new_margin/10.0;

				if ( margin_right < 0 )
					margin_right = 0;
				if ( margin_right > (page_width/2) )
					margin_right = page_width/2 - 0.1;
				sprintf(buf, "%.1f", margin_right);
				gtk_entry_set_text(GTK_ENTRY(right_margin_text), buf);
				break;
		}

	}

	/*
	 * Redraw the screen.
	 */
	PreviewUpdate();

	/*
	 * Update Margin windows.
	 */
	sprintf(buf, "%.3f", margin_top);
	gtk_entry_set_text(GTK_ENTRY(top_margin_text), buf);
	sprintf(buf, "%.3f", margin_bottom);
	gtk_entry_set_text(GTK_ENTRY(bottom_margin_text), buf);
	sprintf(buf, "%.3f", margin_left);
	gtk_entry_set_text(GTK_ENTRY(left_margin_text), buf);
	sprintf(buf, "%.3f", margin_right);
	gtk_entry_set_text(GTK_ENTRY(right_margin_text), buf);

}


/*========================================================================
 *	Name:			ResetFields
 *	Prototype:	static void ResetFields()
 *					
 *	Description:
 *		Reset the dialog to its initial configuration.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ResetFields()
{
	char		buf[32];
	int		i;

	page_width = PAGE_WIDTH_DEFAULT;
	page_height = PAGE_HEIGHT_DEFAULT;

	sprintf(buf, "%.1f", page_width);
	gtk_entry_set_text(GTK_ENTRY(page_width_text), buf);
	sprintf(buf, "%.1f", page_height);
	gtk_entry_set_text(GTK_ENTRY(page_height_text), buf);

	sprintf(buf, "%d", PRINTER_DPI_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(dpi_text), buf);

	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (top_radio_button), TRUE);
	active_margin = BCARDS_TOPMARGIN;

	margin_top = MARGIN_TOP_DEFAULT;			/* Top margin, in inches */
	margin_bottom = MARGIN_BOTTOM_DEFAULT;	/* Bottom margin, in inches */
	margin_left = MARGIN_LEFT_DEFAULT;		/* Left margin, in inches */
	margin_right = MARGIN_RIGHT_DEFAULT;	/* Right margin, in inches */

	sprintf(buf, "%.3f", margin_top);
	gtk_entry_set_text(GTK_ENTRY(top_margin_text), buf);
	sprintf(buf, "%.3f", margin_bottom);
	gtk_entry_set_text(GTK_ENTRY(bottom_margin_text), buf);
	sprintf(buf, "%.3f", margin_left);
	gtk_entry_set_text(GTK_ENTRY(left_margin_text), buf);
	sprintf(buf, "%.3f", margin_right);
	gtk_entry_set_text(GTK_ENTRY(right_margin_text), buf);

	rows = ROWS_DEFAULT;
	columns = COLUMNS_DEFAULT;
	sprintf(buf, "%d", rows);
	gtk_label_set(GTK_LABEL(row_text), buf);
	sprintf(buf, "%d", columns);
	gtk_label_set(GTK_LABEL(column_text), buf);

	/*
	 * Free up and reallocate the grid square selection array.
	 */
	if ( grid_sq != NULL )
		g_free(grid_sq);
	grid_sq = (int *)g_malloc(4*columns*rows);
	for (i=0; i<(rows*columns); i++)
		grid_sq[i] = (int)OFF;

	/*
	 * Reset image menu.
	 */
	image_id = gimp_drawable_image_id(drawable_id);
	GFXRebuildImageMenu();

	/*
	 * Grid/Image Options.
	 */
	sprintf(buf, "%d", SHARPEN_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(sharpen_text), buf);

	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (border_check_button), FALSE);
	border_state = OFF;

	BCardsResetFields();

	/*
	 * Update Page 1 Preview.
	 */
	PreviewUpdate();
}


/*========================================================================
 *	Name:			ArrowCallback
 *	Prototype:	static void ArrowCallback()
 *					
 *	Description:
 *		Handle mouse clicks on arrow buttons.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ArrowCallback(
	GtkWidget	*widget,
	gpointer		data
)
{
	char		buf[32];
	int		i;

	switch((int)data)
	{
		case 1: rows--; if ( rows < 1 ) rows = 1;  break;
		case 2: rows++; if ( rows > 8 ) rows = 8;  break;
		case 3: columns--; if ( columns < 1 ) columns = 1;  break;
		case 4: columns++; if ( columns > 8 ) columns = 8;  break;
	}

	/*
	 * Free up and reallocate the grid square selection array.
	 */
	if ( grid_sq != NULL )
		g_free(grid_sq);
	grid_sq = (int *)g_malloc(4*columns*rows);
	for (i=0; i<(rows*columns); i++)
		grid_sq[i] = (int)OFF;

	sprintf(buf, "%d", rows);
	gtk_label_set(GTK_LABEL(row_text), buf);
	sprintf(buf, "%d", columns);
	gtk_label_set(GTK_LABEL(column_text), buf);

	PreviewUpdate();

	/*
	 * Update grid square margins, in case they've gone "out of bounds".
	 */
	BCardsTextEntryUpdates(NULL, BCARDS_PAGE_TOP_MARGIN_FIELD);
	BCardsTextEntryUpdates(NULL, BCARDS_PAGE_BOTTOM_MARGIN_FIELD);
	BCardsTextEntryUpdates(NULL, BCARDS_PAGE_LEFT_MARGIN_FIELD);
	BCardsTextEntryUpdates(NULL, BCARDS_PAGE_RIGHT_MARGIN_FIELD);
}


/*========================================================================
 *	Name:			ComputeSuggestedSize
 *	Prototype:	static void ComputeSuggestedSize()
 *					
 *	Description:
 *		Compute the recommended size that the source image should be based on
 *		the size of the page, the margin settings and the number of cards on
 *		the page.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ComputeSuggestedSize(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	new_page_width;
	gfloat	new_page_height;
	int		dpi;
	gfloat	new_sug_width;
	gfloat	new_sug_height;
	gfloat	grid_width, grid_height;
	char		buf[32];

	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));

	new_page_width = page_width - margin_left - margin_right;
	new_page_height = page_height - margin_top - margin_bottom;

	grid_width = new_page_width / columns;
	grid_height = new_page_height / rows;

	new_sug_width = 
			(int)((grid_width - imargin_left - imargin_right) * (gfloat)dpi);
	new_sug_height = 
			(int)((grid_height - imargin_top - imargin_bottom) * (gfloat)dpi);

	sprintf(buf, "%dx%d", (int)new_sug_width, (int)new_sug_height);
	gtk_label_set(GTK_LABEL(src_suggested_size_text), buf);
}


/*========================================================================
 *	Name:			TextEntryUpdates
 *	Prototype:	static void TextEntryUpdates()
 *					
 *	Description:
 *		Update the display based on input from the user.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
TextEntryUpdates(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch((int)data)
	{
		case BCARDS_PAGE_WIDTH_FIELD:
			page_width = atof(gtk_entry_get_text(GTK_ENTRY(page_width_text)));

			/*
			 * Make sure we're not too small or large and that we don't allow
			 * the margins to overlap.
			 */
			if ( page_width > PREVIEW_WSIZE / 10 )
				page_width = PREVIEW_WSIZE / 10;
	
			if ( page_width < (margin_left+margin_right) )
				page_width = margin_left+margin_right;
			break;

		case BCARDS_PAGE_HEIGHT_FIELD:
			page_height = atof(gtk_entry_get_text(GTK_ENTRY(page_height_text)));

			/*
			 * Make sure we're not too small or large and that we don't allow
			 * the margins to overlap.
			 */
			if ( page_height > PREVIEW_HSIZE / 10 )
				page_height = PREVIEW_HSIZE / 10;
			if ( page_height < (margin_top+margin_bottom) )
				page_height = margin_top+margin_bottom;
			break;

		case BCARDS_PAGE_TOP_MARGIN_FIELD:
			margin_top = atof(gtk_entry_get_text(GTK_ENTRY(top_margin_text)));
			if ( margin_top > page_height/2 )
				margin_top = page_height/2;
			if ( margin_top < 0 )
				margin_top = 0;
			break;

		case BCARDS_PAGE_BOTTOM_MARGIN_FIELD:
			margin_bottom = atof(gtk_entry_get_text(GTK_ENTRY(bottom_margin_text)));
			if ( margin_bottom > page_height/2 )
				margin_bottom = page_height/2;
			if ( margin_bottom < 0 )
				margin_bottom = 0;
			break;

		case BCARDS_PAGE_LEFT_MARGIN_FIELD:
			margin_left = atof(gtk_entry_get_text(GTK_ENTRY(left_margin_text)));
			if ( margin_left > page_width/2 )
				margin_left = page_width/2;
			if ( margin_left < 0 )
				margin_left = 0;
			break;

		case BCARDS_PAGE_RIGHT_MARGIN_FIELD:
			margin_right = atof(gtk_entry_get_text(GTK_ENTRY(right_margin_text)));
			if ( margin_right > page_width/2 )
				margin_right = page_width/2;
			if ( margin_right < 0 )
				margin_right = 0;
			break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			OKCallback
 *	Prototype:	static void OKCallback()
 *					
 *	Description:
 *		Process the image.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	1. Validate input
 * 2. Create a duplicate of the original image.
 * 3. Flatten the duplicate.
 * 4. Scale it to the size of a single business card on our printed page.
 * 5. Select and copy the contents of the scaled image duplicate.
 * 6. Create a new page the correct dimenions of the page.
 * 7. For each row/column, paste and position the scaled image duplicate.
 * 8. Delete the duplicate.
 * 9. Display the new page.
 *
 *	Notes:
 * This isn't very efficient - it would be better to block copy the pixel
 * regions from the copy to the new image, but this method was quick and
 * easy to do.  Future revisions will worry about speed.
 *========================================================================*/
static void
OKCallback()
{
	gint32			new_image_id=0, copy_image_id;
	gint32			new_layer_id;
	gint32			selection_type;
	int				dpi;
	int				sharpen;
	int				new_image_width, new_image_height;
	int				copy_image_width, copy_image_height;
	gfloat			new_page_width, new_page_height;
	gfloat			grid_width, grid_height;
	gfloat			new_sug_width, new_sug_height;
	int				i,j;
	int				x_offset, y_offset;
	GimpDrawable		*new_drawable_id;
	GimpPixelRgn		pixel_rgn, copy_pixel_rgn;
	unsigned char	*pixel_block, *copy_pixel_block;
	GimpParam			*return_vals;
	int				nreturn_vals;
	char				buf[128];
	gint32			*images;
	gint				nimages;
	gboolean			found;
	gint32			active_drawable;
	gint32			*layers;

	/*
	 * Validate the image id - it may have gone away since the time we
	 * last updated the menu.
	 */
	images = gimp_image_list(&nimages);
	found = FALSE;
	if ( images != NULL )
	{
		for (i=0; i<nimages; i++ )
		{
			if ( images[i] == image_id )
				found = TRUE;
		}
		g_free(images);
	}

	if (found == FALSE)
	{
		sprintf(buf, "%s %s",
			"The selected image seems to have been deleted.  ",
			"Please select another image.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);

		/*
		 * Rebuild the images menu.
		 */
		GFXRebuildImageMenu();

		return;
	}

/*
	return_vals = gimp_run_procedure ("gimp_image_active_drawable",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	active_drawable = return_vals[1].data.d_drawable;
	gimp_destroy_params (return_vals, nreturn_vals);
*/

	/*
	 * Verify at least one grid square has been selected.
	 */
	found = FALSE;
	for (i=0; i<(rows*columns); i++)
	{
		if (grid_sq[i] == (int)ON)
			found = TRUE;
	}
	
	if (found == FALSE)
	{
		sprintf(buf, 
			"You have not selected any grid squares.\n"
			"Please select one or more grid squares\n"
			"by clicking the right mouse button on them\n"
			"or clicking on the Select All button.\n"
		);
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 300, 60);

		return;
	}

	/*
	 * Disable the undo buffer.  We only need to do this for the duplication
	 * process since the rest of the operations are performed on different
	 * image IDs.
	 */
	gimp_image_undo_disable(image_id);

	/*
	 * If all the data was valid, then create a duplicate image of the
	 * source image.
	 */
	return_vals = gimp_run_procedure ("gimp_channel_ops_duplicate",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);

	if (return_vals[0].data.d_status == GIMP_PDB_SUCCESS)
		copy_image_id = return_vals[1].data.d_int32;
	else
	{
		gimp_destroy_params (return_vals, nreturn_vals);
		gimp_image_undo_enable(image_id);
		return;
	}
	gimp_destroy_params (return_vals, nreturn_vals);
	gimp_image_undo_enable(image_id);


	/*
	 * If necessary, flatten the duplicate.
	 */
	GFXMsgWindowUpdate(
		"Flattening Image...\n",
		msg_data);
	gimp_image_flatten(copy_image_id);

	/*
	 * Scale the duplicate to the size appropriate for a single card
	 * based on the page size, margins and number of cards per page.
	 */
	GFXMsgWindowUpdate(
		"Scaling to appropriate size...\n",
		msg_data);
	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));

	new_page_width = page_width - margin_left - margin_right;
	new_page_height = page_height - margin_top - margin_bottom;
	grid_width = new_page_width / columns;
	grid_height = new_page_height / rows;
	new_sug_width = (grid_width - imargin_left - imargin_right) * (gfloat)dpi;
	new_sug_height = (grid_height - imargin_top - imargin_bottom) * (gfloat)dpi;

	return_vals = gimp_run_procedure ("gimp_image_scale",
							&nreturn_vals,
							GIMP_PDB_IMAGE, copy_image_id,
							GIMP_PDB_INT32, (int)new_sug_width,
							GIMP_PDB_INT32, (int)new_sug_height,
							GIMP_PDB_END);
	if (return_vals[0].data.d_status != GIMP_PDB_SUCCESS)
	{
		return_vals = gimp_run_procedure ("gimp_image_delete",
							&nreturn_vals,
							GIMP_PDB_IMAGE, copy_image_id,
							GIMP_PDB_END);
		gimp_destroy_params (return_vals, nreturn_vals);
		GFXMsgWindowUpdate(
			"Scaling to appropriate size...\n",
			"Error scaling window - CardGFX failed!",
			msg_data);
		return;
	}
	gimp_destroy_params (return_vals, nreturn_vals);


	/*
	 * Sharpen the image but only if:
	 * 1. the sharpen plug-in is available
	 * 2. the sharpen value is > 0.
	 */
	if ( sharpen_text != NULL )
	{
		sharpen = atoi(gtk_entry_get_text(GTK_ENTRY(sharpen_text)));
		if ( sharpen > 0 )
		{
			GFXMsgWindowUpdate("Sharpening copy...\n", msg_data);

			/*
			 * Make the only layer in the copy image the active layer.
			 */
			layers = gimp_image_get_layers(copy_image_id, &nreturn_vals);
			gimp_image_set_active_layer(copy_image_id, layers[0]);
			if ( layers != NULL )
				g_free(layers);

			/*
			 * Get the active drawable for the active layer.  We need to
			 * pass this to the Sharpen plug-in.
			 */
			return_vals = gimp_run_procedure ("gimp_image_active_drawable",
							&nreturn_vals,
							GIMP_PDB_IMAGE, copy_image_id,
							GIMP_PDB_END);
			active_drawable = return_vals[1].data.d_drawable;
			gimp_destroy_params (return_vals, nreturn_vals);

			/*
			 * Sharpen the image by the specified amount.
			 */
			return_vals = gimp_run_procedure ("plug_in_sharpen",
							&nreturn_vals,
							GIMP_PDB_INT32, GIMP_RUN_NONINTERACTIVE,
							GIMP_PDB_IMAGE, copy_image_id,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, sharpen,
							GIMP_PDB_END);
			if (return_vals[0].data.d_status != GIMP_PDB_SUCCESS)
			{
				switch ( return_vals[0].data.d_status )
				{
					case GIMP_PDB_EXECUTION_ERROR:
						printf("sharpen: GIMP_PDB_EXECUTION_ERROR\n");
						break;
					case GIMP_PDB_CALLING_ERROR:
						printf("sharpen: GIMP_PDB_CALLING_ERROR\n");
						break;
					default:
						printf("sharpen: unknown status\n");
						break;
				}
			}
			gimp_destroy_params (return_vals, nreturn_vals);
		}
	}


	/*
	 * Create a new image window of the appropriate size.  Add a blank,
	 * white layer to it.
	 */
	GFXMsgWindowUpdate(
		"Creating New Image Window...\n",
		msg_data);
	new_image_width = page_width  * dpi;
	new_image_height = page_height * dpi;

	new_image_id = gimp_image_new(new_image_width, new_image_height, 0);

	new_layer_id = gimp_layer_new(
							new_image_id,		/* Image ID */
							"Background",		/* layer name */
							new_image_width,	/* layer width */
							new_image_height,	/* layer height */
							0,						/* Image type = RGB */
							100,					/* layer opacity */
							0);					/* Blend mode = Normal */
	gimp_image_add_layer (new_image_id, new_layer_id, 0);
	new_drawable_id = gimp_drawable_get(new_layer_id);

	/* set entire new layer to white */
	gimp_pixel_rgn_init (
		&pixel_rgn, 
		new_drawable_id, 
		0, 
		0, 
		new_drawable_id->width, 
		new_drawable_id->height, 
		TRUE, 
		FALSE); 
	pixel_block = 
		(unsigned char *)g_malloc (new_image_width * new_image_height * 3);
	memset(pixel_block, 0xff, new_image_width*new_image_height*3);
	gimp_pixel_rgn_set_rect (
		&pixel_rgn, 
		pixel_block, 
		0, 
		0, 
		new_image_width, 
		new_image_height);
	g_free(pixel_block);


	/*
	 * Save the duplicate image's contents - we'll be using them in a 
	 * pseudo-paste operation in a moment.
	 */
	new_drawable_id = 
		gimp_drawable_get( gimp_image_get_active_layer(copy_image_id) );

	copy_image_width = new_drawable_id->width;
	copy_image_height = new_drawable_id->height;
	gimp_pixel_rgn_init (
			&copy_pixel_rgn, 
			new_drawable_id, 
			0, 
			0, 
			new_drawable_id->width, 
			new_drawable_id->height, 
			TRUE, 
			FALSE); 

	copy_pixel_block = 
		(unsigned char *)g_malloc(copy_image_width * copy_image_height * 3);

	gimp_pixel_rgn_get_rect (
			&copy_pixel_rgn, 
			copy_pixel_block, 
			0, 
			0, 
			new_drawable_id->width, 
			new_drawable_id->height);

	/*
	 * Now for each column of each row of the page, copy the duplicate 
	 * into the new image window and position it accordingly.
	 */
	for (i=0; i<rows; i++)
	{
		for (j=0; j<columns; j++)
		{
			if ( grid_sq[(i*columns)+j] == (int)ON )
			{
				sprintf(buf,"Copying into grid square %dx%d...\n", i,j);
				GFXMsgWindowUpdate(
					buf,
					msg_data);

				x_offset = 
					(int)((j*new_sug_width) + 
							(dpi * (margin_left + 
							((j+1)*imargin_left) + (j*imargin_right) )));

				y_offset = 
					(int)((i*new_sug_height) +
							(dpi * (margin_top +
							((i+1)*imargin_top)+ (i*imargin_bottom) )));

				gimp_pixel_rgn_set_rect (
					&pixel_rgn, 
					copy_pixel_block, 
					x_offset,
					y_offset,
					copy_image_width, 
					copy_image_height);

			}
			else
			{
				sprintf(buf,"Skipping grid square %dx%d...\n", i,j);
				GFXMsgWindowUpdate(
					buf,
					msg_data);
			}
		}
	}
	g_free(copy_pixel_block);


	/*
	 * Delete the duplicate image.
	 */
	return_vals = gimp_run_procedure ("gimp_image_delete",
							&nreturn_vals,
							GIMP_PDB_IMAGE, copy_image_id,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Add borders to image, if requested.
	 */
	if ( border_state == ON )
	{
		sprintf(buf,"Adding borders...\n");
		GFXMsgWindowUpdate(
			buf,
			msg_data);

		selection_type = 2;
		for (i=0; i<rows; i++)
		{
			for (j=0; j<columns; j++)
			{
				if ( grid_sq[(i*columns)+j] == (int)ON )
				{
					x_offset = 
						(int)((j*new_sug_width) + 
							(dpi * (margin_left + 
							((j+1)*imargin_left) + (j*imargin_right) )));

					y_offset = 
						(int)((i*new_sug_height) +
							(dpi * (margin_top +
							((i+1)*imargin_top)+ (i*imargin_bottom) )));

					/*
					 * Create/add a selection to new image.
					 */
					return_vals = gimp_run_procedure ("gimp_rect_select",
									&nreturn_vals,
									GIMP_PDB_IMAGE, new_image_id,
									GIMP_PDB_FLOAT, (gfloat)x_offset,
									GIMP_PDB_FLOAT, (gfloat)y_offset,
									GIMP_PDB_FLOAT, (gfloat)copy_image_width,
									GIMP_PDB_FLOAT, (gfloat)copy_image_height,
									GIMP_PDB_INT32, selection_type,
									GIMP_PDB_INT32, 0,
									GIMP_PDB_FLOAT, 0.0,
									GIMP_PDB_END);
					gimp_destroy_params (return_vals, nreturn_vals);
					selection_type = 0;
				}
			}
		}

		/*
		 * Get the active drawable for the new image.
		 */
		return_vals = gimp_run_procedure ("gimp_image_active_drawable",
							&nreturn_vals,
							GIMP_PDB_IMAGE, new_image_id,
							GIMP_PDB_END);
		active_drawable = return_vals[1].data.d_drawable;
		gimp_destroy_params (return_vals, nreturn_vals);

		/*
		 * Stroke selections.
		 */
		return_vals = gimp_run_procedure ("gimp_edit_stroke", 
							&nreturn_vals,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_END);
		gimp_destroy_params (return_vals, nreturn_vals);
		return_vals = gimp_run_procedure ("gimp_selection_clear", 
							&nreturn_vals,
							GIMP_PDB_IMAGE, new_image_id,
							GIMP_PDB_END);
		gimp_destroy_params (return_vals, nreturn_vals);
	}


	/*
	 * Display the new image, the one that should be printed.
	 */
	gimp_display_new (new_image_id);
	gimp_displays_flush ();

	/*
	 * Thats it - say b'bye!
	CloseCallback();
	 */
}


/*========================================================================
 *	Name:			GridSelect
 *	Prototype:	static void GridSelect()
 *					
 *	Description:
 *		Turn on or off all grid squares.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
GridSelect(
	GtkWidget	*widget,
	gpointer		data
)
{
	int	i;

	switch ((int)data)
	{
		case ON:
			for (i=0; i<(rows*columns); i++)
				grid_sq[i] = (int)ON;
			break;

		case OFF:
			for (i=0; i<(rows*columns); i++)
				grid_sq[i] = (int)OFF;
			break;
	}

	PreviewUpdate();
}
