/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: image.c
 *            DESCRIPTION: image configuration routines
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define IMAGE_C		/* needed by debug library */

/* === System Headers === */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "bcards.h"
#include "gfxcommon.h"
#include "debug.h"


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();


/* === Public routine prototypes === */
void BCardsImageSetup();
void BCardsImageSelect();
void BCardsResetFields();
void BCardsTextEntryUpdates();
void GFXRebuildImageMenu();


/* === Private routine prototypes === */
static gint ImageConstraints();
static void SharpenUpdate();
static void SharpenTextUpdate();
static void PreviewUpdate();
static void PreviewMotion();
static void PreviewButtonPress();
static void PreviewButtonRelease();
static void MarginToggle();
static void BorderToggle();


/* === Global Variables === */
extern GtkWidget	*src_current_size_text;
extern GtkWidget	*dpi_text;
extern int			image_id;	/* ID of image to use as source */
extern int			image_width, image_height;
extern gfloat		page_width;
extern gfloat		page_height;
extern gfloat		margin_top;
extern gfloat		margin_bottom;
extern gfloat		margin_left;
extern gfloat		margin_right;
extern int			columns;
extern int			rows;

GtkWidget			*sharpen_text=NULL;
GtkWidget			*border_check_button;

gfloat	imargin_top = IMARGIN_TOP_DEFAULT;	
gfloat	imargin_bottom = IMARGIN_BOTTOM_DEFAULT;
gfloat	imargin_left = IMARGIN_LEFT_DEFAULT;
gfloat	imargin_right = IMARGIN_RIGHT_DEFAULT;

int		border_state = OFF;

/* === Static Variables === */
static GtkWidget			*image_menu, *image_options;
static GtkDrawingArea	*preview;
static GtkWidget			*sharpen_scale;
static GtkObject			*sharpen_adj;
static GtkWidget			*itop_margin_text;
static GtkWidget			*ibottom_margin_text;
static GtkWidget			*ileft_margin_text;
static GtkWidget			*iright_margin_text;
static GtkWidget			*top_radio_button;

static int		active_margin=BCARDS_TOPMARGIN;
static int		mouse_x, mouse_y;	/* current location of mouse on mouse clicks */
static int		drag_state;



/*========================================================================
 *	Name:			BCardsImageSetup()
 *	Prototype:	static void BCardsImageSetup()
 *					
 *	Description:
 *		Image configuration components.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *========================================================================*/
void
BCardsImageSetup(
	GtkWidget	*notebook
)
{
	GtkWidget		*vbox;
	GtkWidget		*table;
	GtkWidget		*label;
	GtkWidget		*frame;
	GtkWidget		*hbox;
	GtkWidget		*menu, *option_menu;
	GtkWidget		*margins_table;
	GtkWidget		*toggle;
	GSList			*group = NULL;

	char				**proc_names;
	int				nitems;
	char				buf[32];

	/*
	 * The table in which the Business Cards components will go.
	 */
	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(table), 8);
	gtk_widget_show(table);

	/*
	 * Add the table to page 1 of the notebook.
	 */
	label = gtk_label_new("Grid/Image Options");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table, label);

	/*
	 * A small preview window to show the columns/rows/margins setup.
	 */
	frame = gtk_frame_new("Adjust Grid Square Margins");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 3, 0, 0, 0, 0);
	gtk_widget_show(frame);
	preview = (GtkDrawingArea *)gtk_drawing_area_new();
	gtk_drawing_area_size(preview, IMAGE_PREVIEW_WSIZE, IMAGE_PREVIEW_HSIZE);
	gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *)preview);
	gtk_widget_show((GtkWidget *)preview);

	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "expose_event",
			(GtkSignalFunc)PreviewUpdate,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_press_event",
			(GtkSignalFunc)PreviewButtonPress,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_release_event",
			(GtkSignalFunc)PreviewButtonRelease,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "motion_notify_event",
			(GtkSignalFunc)PreviewMotion,
			NULL);

	gtk_widget_set_events((GtkWidget *)preview,
		GDK_EXPOSURE_MASK | GDK_BUTTON_MOTION_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
 
	/*
	 * Margins information go in another table, one that sits
	 * in the middle row of the main table.
	 */
	frame = gtk_frame_new("Grid Square Margins (in inches)");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);

	margins_table = gtk_table_new(2, 6, FALSE);
	gtk_container_border_width(GTK_CONTAINER(margins_table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(margins_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(margins_table), 8);
	gtk_container_add (GTK_CONTAINER (frame), margins_table);
	gtk_widget_show(margins_table);

	label = gtk_label_new("Top");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_set_usize( GTK_WIDGET(label), 45, -1 );
	gtk_widget_show(label);
	
	itop_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", imargin_top);
	gtk_entry_set_text(GTK_ENTRY(itop_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (itop_margin_text), "changed",
		(GtkSignalFunc) BCardsTextEntryUpdates, 
		(gpointer)BCARDS_PAGE_TOP_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)itop_margin_text, 
		2, 3, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(itop_margin_text);

	label = gtk_label_new("Bottom");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	ibottom_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", imargin_bottom);
	gtk_entry_set_text(GTK_ENTRY(ibottom_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (ibottom_margin_text), "changed",
		(GtkSignalFunc) BCardsTextEntryUpdates, 
		(gpointer)BCARDS_PAGE_BOTTOM_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)ibottom_margin_text, 
		2, 3, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(ibottom_margin_text);

	label = gtk_label_new("Left");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		4, 5, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_set_usize( GTK_WIDGET(label), 45, -1 );
	gtk_widget_show(label);
	
	ileft_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", imargin_left);
	gtk_entry_set_text(GTK_ENTRY(ileft_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (ileft_margin_text), "changed",
		(GtkSignalFunc) BCardsTextEntryUpdates, 
		(gpointer)BCARDS_PAGE_LEFT_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)ileft_margin_text, 
		5, 6, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(ileft_margin_text);

	label = gtk_label_new("Right");
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)label, 
		4, 5, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(label);
	
	iright_margin_text = gtk_entry_new_with_max_length(6);
	sprintf(buf, "%.3f", imargin_right);
	gtk_entry_set_text(GTK_ENTRY(iright_margin_text), buf);
	gtk_signal_connect (GTK_OBJECT (iright_margin_text), "changed",
		(GtkSignalFunc) BCardsTextEntryUpdates, 
		(gpointer)BCARDS_PAGE_RIGHT_MARGIN_FIELD);
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)iright_margin_text, 
		5, 6, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(iright_margin_text);

	/*
	 * Adjust width of all text input fields
	 */
	gtk_widget_set_usize( iright_margin_text, 50, -1 );
	gtk_widget_set_usize( ileft_margin_text, 50, -1 );
	gtk_widget_set_usize( ibottom_margin_text, 50, -1 );
	gtk_widget_set_usize( itop_margin_text, 50, -1 );

	/*
	 * Radio buttons for each of the margins - these determine which lines
	 * move in the preview.
	 */
	top_radio_button = toggle = gtk_radio_button_new(NULL);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	active_margin = BCARDS_TOPMARGIN;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_TOPMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_BOTTOMMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		3, 4, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_LEFTMARGIN);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new(group);
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_table_attach(GTK_TABLE(margins_table), (GtkWidget *)toggle, 
		3, 4, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) MarginToggle, (gpointer) BCARDS_RIGHTMARGIN);
	gtk_widget_show (toggle);


	/*
	 * Vertical Box for sharpen and border options.
	 */
	vbox = gtk_vbox_new(FALSE, 8);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)vbox, 
		1, 2, 1, 3, 0, 0, 0, 0);
	gtk_widget_show(vbox);
	

	/*
	 * Add Sharpen options if available.
	 */
	gimp_procedural_db_query( "plug.in.sharpen", ".*", ".*", ".*", ".*", ".*", ".*",
		&nitems, &proc_names);
	if ( nitems > 0 )
	{
		frame = gtk_frame_new("Sharpen Amount");
		gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
		gtk_box_pack_start (GTK_BOX (vbox), frame, TRUE, TRUE, 0);
		gtk_widget_show(frame);
	
		hbox = gtk_hbox_new(FALSE, 4);
		gtk_container_add (GTK_CONTAINER (frame), hbox);
		gtk_widget_show(hbox);
	
		sharpen_text = gtk_entry_new_with_max_length(4);
		gtk_box_pack_start (GTK_BOX (hbox), sharpen_text, TRUE, TRUE, 0);
		sprintf(buf, "%d", SHARPEN_DEFAULT);
		gtk_entry_set_text(GTK_ENTRY(sharpen_text), buf);
		gtk_signal_connect(GTK_OBJECT(sharpen_text), "changed",
			(GtkSignalFunc)SharpenTextUpdate, NULL);
		gtk_widget_show(sharpen_text);
	
		sharpen_adj = 
			gtk_adjustment_new(SHARPEN_DEFAULT, 0.0, 110.0, 1.0, 10.0, 10.0);
		sharpen_scale = gtk_hscale_new(GTK_ADJUSTMENT(sharpen_adj));
		gtk_box_pack_start (GTK_BOX (hbox), sharpen_scale, TRUE, TRUE, 0);
		gtk_widget_set_usize( sharpen_scale, 155, -1 );
		gtk_widget_show(sharpen_scale);
	
		gtk_signal_connect(GTK_OBJECT(sharpen_adj), "value_changed",
			(GtkSignalFunc)SharpenUpdate, NULL);

		gtk_widget_set_usize( sharpen_text, 50, -1 );
		gtk_widget_set_usize( sharpen_scale, 135, -1 );
	}

	/*
	 * A toggle button for adding a solid border around the image, along
	 * the specified margins.
	 */
	border_check_button = toggle = 
		gtk_check_button_new_with_label("Add Border To Margin Lines");
	gtk_box_pack_start (GTK_BOX (vbox), toggle, TRUE, FALSE, 8);
	gtk_widget_set_usize( toggle, 185, -1 );
	border_state = OFF;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) BorderToggle, (gpointer) NULL);
	gtk_widget_show (toggle);

	/*
	 * Image Windows menu - used to select which image to use when creating
	 * the full page of cards.
	 */
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)hbox, 
		0, 2, 3, 4, 0, 0, 0, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new("Source Image");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, FALSE, 8);
	gtk_widget_show(label);

	image_options = option_menu = gtk_option_menu_new();
	gtk_box_pack_start (GTK_BOX (hbox), option_menu, TRUE, TRUE, 8);
	gtk_widget_set_usize( GTK_WIDGET(image_options), 220, -1 );
	gtk_widget_show(option_menu);

	image_menu = menu = gimp_image_menu_new(ImageConstraints,
					BCardsImageSelect,
					NULL,
					image_id);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);

}


/*========================================================================
 *	Name:			ImageConstraints
 *	Prototype:	static gint ImageConstraints()
 *					
 *	Description:
 *		Callback used to determine which image windows we'll include in our
 *		menu of images.  Basically, we include all of them.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static gint
ImageConstraints(
	gint32	image_id, 
	gint32	drawable_id, 
	gpointer	data
)
{
	if (image_id == -1)
		return TRUE;
 
	/* No real constraints at this point */
	return TRUE;
}


/*========================================================================
 *	Name:			BCardsImageSelect
 *	Prototype:	void BCardsImageSelect()
 *					
 *	Description:
 *		What to do when the user selects a particular image from the menu.
 *		Basically, just save the id of that image and update the current
 *		size display.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
BCardsImageSelect(
	gint32 	id, 
	gpointer	data
)
{
#ifdef DEBUG
	char					*fname="BCardsImageSelect()";
#endif

	gint32				*images;
	gint					nimages;
	char					buf[32];
	gboolean				found;
	int					i;

	static gboolean	firsttime = TRUE;

	DBGEnter();

	DBGPrintf(DBG_INFO,
		("src_current_size_text: 0x%08x\n", src_current_size_text));
	DBGPrintf(DBG_INFO, ("image id: %d\n", id));

	/*
	 * Validate the image id - it may have gone away since the time we
	 * last updated the menu.
	 */
	images = gimp_image_list(&nimages);
	found = FALSE;
	if ( images != NULL )
	{
		for (i=0; i<nimages; i++ )
		{
			if ( images[i] == image_id )
				found = TRUE;
		}
		g_free(images);
	}

	if (found == FALSE)
	{
		sprintf(buf, "%s %s",
			"The selected image seems to have been deleted.  ",
			"Please select another image.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);

		/*
		 * Rebuild the images menu.
		 */
		GFXRebuildImageMenu();

		return;
	}

/*
	if ( ( src_current_size_text != NULL ) && ( id != 0 ) )
*/
	if ( src_current_size_text != NULL )
	{
		image_id = id;

		image_width  = gimp_image_width(image_id);
		image_height = gimp_image_height(image_id);

		sprintf(buf, "%dx%d", image_width, image_height);
		gtk_label_set_text(GTK_LABEL(src_current_size_text), buf);

		if ( !firsttime )
			BCardsResetFields();
	}

	firsttime = FALSE;

}


/*========================================================================
 *	Name:			GFXRebuildImageMenu
 *	Prototype:	void GFXRebuildImageMenu()
 *					
 *	Description:
 *		Rebuild the image options menu.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
GFXRebuildImageMenu()
{
	gtk_widget_hide(image_options);
	gtk_widget_destroy(image_menu);
	image_menu = gimp_image_menu_new(ImageConstraints,
					BCardsImageSelect,
					NULL,
					image_id);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(image_options), image_menu);
	gtk_widget_show(image_options);
}


/*========================================================================
 *	Name:			SharpenUpdate
 *	Prototype:	static void SharpenUpdate()
 *					
 *	Description:
 *		Handle updates from Sharpen scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
SharpenUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	sprintf(buf, "%.0f", adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(sharpen_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(sharpen_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(sharpen_text), NULL);

}


/*========================================================================
 *	Name:			SharpenTextUpdate
 *	Prototype:	static void SharpenTextUpdate()
 *					
 *	Description:
 *		Handle updates from Sharpen text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
SharpenTextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(sharpen_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(sharpen_adj)->upper))
	{
		GTK_ADJUSTMENT(sharpen_adj)->value = newvalue;
		gtk_signal_emit_by_name(sharpen_adj, "value_changed");
	}

}


/*========================================================================
 *	Name:			PreviewUpdate
 *	Prototype:	static void PreviewUpdate()
 *					
 *	Description:
 *		Redraw the preview window with margin lines
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewUpdate()
{
	int				draw_width, draw_height;
	int				image_width, image_height;
	int				dpi;

	int				xoffset, yoffset;
	int				top_offset, bottom_offset;
	int				left_offset, right_offset;
	gfloat			sfactor;
	static GdkGC	*gcWhite = NULL;
	static GdkGC	*gcBlack = NULL;
	static GdkGC	*gcDotted = NULL;

	static GdkPixmap	*offscreen_pixmap=NULL;

	if ( preview->widget.window == NULL )
		return;

	/*
	 * Generate an offscreen pixmap.  We draw in this and then copy it to
	 * the real window in order to eliminate flashing - thats called double
	 * buffering.
	 */
	if ( offscreen_pixmap == NULL )
	{
		offscreen_pixmap = 
			gdk_pixmap_new(
				GTK_WIDGET(preview)->window,
				GTK_WIDGET(preview)->allocation.width,
				GTK_WIDGET(preview)->allocation.height,
				-1);
	}

	/*
	 * Clear the offscreen pixmap.
	 */
	gdk_draw_rectangle(
		offscreen_pixmap,
		GTK_WIDGET(preview)->style->bg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		TRUE,
		0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height);


	if (gcWhite == NULL)
	{
		gcWhite = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcWhite, GTK_WIDGET(preview)->style->white_gc );
		gcBlack = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcBlack, GTK_WIDGET(preview)->style->black_gc );
		gcDotted = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcDotted, GTK_WIDGET(preview)->style->black_gc );
	}

	/*
	 * Compute the scaling factor needed to convert the margings into pixels
	 */
	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));

	/*
	 * Compute the image size with respect to the drawing areas physical
	 * region.
	 */
	image_width = (page_width - margin_left - margin_right)/columns * dpi;
	image_height = (page_height - margin_top - margin_bottom)/rows  * dpi;

	if ( image_width > image_height )
		sfactor = (gfloat)(IMAGE_PREVIEW_WSIZE-1) / (gfloat)image_width;
	else
		sfactor = (gfloat)(IMAGE_PREVIEW_HSIZE-1) / (gfloat)image_height;

	draw_width = sfactor * image_width;
	draw_height = sfactor * image_height;

	xoffset = (IMAGE_PREVIEW_WSIZE - draw_width) / 2;
	yoffset = (IMAGE_PREVIEW_HSIZE - draw_height) / 2;


	/*
	 * Draw page.
	 */
	gdk_draw_rectangle(offscreen_pixmap, gcWhite, 1,
		xoffset, yoffset, draw_width, draw_height);

	/*
	 * Draw margins.  One of these will be active from the toggles, so it
	 * gets drawn in red.  All others are drawn in black.
	 */
	top_offset = yoffset + (imargin_top*72);
	bottom_offset = yoffset + draw_height - (imargin_bottom*72);
	left_offset = xoffset + (imargin_left*72);
	right_offset = xoffset + draw_width - (imargin_right*72);

	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Top margin line */
	if ( active_margin == BCARDS_TOPMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		0, top_offset, IMAGE_PREVIEW_WSIZE, top_offset );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Bottom margin line */
	if ( active_margin == BCARDS_BOTTOMMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		0, bottom_offset, IMAGE_PREVIEW_WSIZE, bottom_offset );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Left margin line */
	if ( active_margin == BCARDS_LEFTMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		left_offset, 0, left_offset, IMAGE_PREVIEW_HSIZE );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/* Right margin line */
	if ( active_margin == BCARDS_RIGHTMARGIN )
		gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);
	gdk_draw_line(offscreen_pixmap, gcDotted, 
		right_offset, 0, right_offset, IMAGE_PREVIEW_HSIZE );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);

	/*
	 * Copy pixmap into the preview drawing area.
	 */
	gdk_draw_pixmap(
		GTK_WIDGET(preview)->window,
		GTK_WIDGET(preview)->style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(preview))],
		offscreen_pixmap,
		0,0, 0,0,
		GTK_WIDGET(preview)->allocation.width,
		GTK_WIDGET(preview)->allocation.height
		);

	/*
	 * Force screen updates.
	 */
	gdk_flush();

}


/*========================================================================
 *	Name:			PreviewButtonPress
 *	Prototype:	static void PreviewButtonPress()
 *					
 *	Description:
 *		Determine mouse coordinates when the mouse button is pressed
 *		in the drawing area (preview) widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonPress(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * If first button was pressed, we're going to be handling mouse drags
	 * to change page size or margin adjustments.
	 */
	if ( event->button == 1 )
		drag_state = ON;
	else
		drag_state = OFF;

	/*
	 * Update the preview window.
	 */
	PreviewUpdate();
}


/*========================================================================
 *	Name:			PreviewButtonRelease
 *	Prototype:	static void PreviewButtonRelease()
 *					
 *	Description:
 *		Reset options when the mouse button is released.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonRelease(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	drag_state = OFF;
}


/*========================================================================
 *	Name:			PreviewMotion
 *	Prototype:	static void PreviewMotion()
 *					
 *	Description:
 *		Update preview window settings based on which margin line
 *		is going to be moved.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewMotion(
	GtkWidget		*widget,
	GdkEventMotion	*event
)
{
	int		offsetx, offsety;
	gfloat	new_margin;
	char		buf[32];
	gfloat	grid_width, grid_height;
	gfloat	sfactor;
	int		dpi;

	if ( drag_state == OFF )
		return;

	/*
	 * Get current mouse location and compute its offset from when the mouse
	 * button was pressed.
	 */
	offsetx = event->x - mouse_x;
	offsety = event->y - mouse_y;

	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * Determine current grid size.
	 */
	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));
	grid_width = (page_width - margin_left - margin_right)/columns * dpi;
	grid_height = (page_height - margin_top - margin_bottom)/rows * dpi;
	if ( grid_width > grid_height )
		sfactor = (gfloat)(IMAGE_PREVIEW_WSIZE-1) / (gfloat)grid_width;
	else
		sfactor = (gfloat)(IMAGE_PREVIEW_HSIZE-1) / (gfloat)grid_height;

	/*
	 * Calculate new margin position.
	 */
	switch (active_margin)
	{
		case BCARDS_TOPMARGIN:
			new_margin = imargin_top * 72.0 + (0.5*offsety);
			imargin_top = new_margin/72.0;

			if ( imargin_top > (sfactor*grid_height/72)/2 )
			{
				imargin_top = (sfactor*grid_height/72)/2;
			}
			if ( imargin_top < 0 )
			{
				imargin_top = 0;
			}
			sprintf(buf, "%.3f", imargin_top);
			gtk_entry_set_text(GTK_ENTRY(itop_margin_text), buf);
			break;

		case BCARDS_BOTTOMMARGIN:
			new_margin = imargin_bottom * 72.0 - (0.5*offsety);
			imargin_bottom = new_margin/72.0;

			if ( imargin_bottom > (sfactor*grid_height/72)/2 )
			{
				imargin_bottom = (sfactor*grid_height/72)/2;
			}
			if ( imargin_bottom < 0 )
			{
				imargin_bottom = 0;
			}
			sprintf(buf, "%.3f", imargin_bottom);
			gtk_entry_set_text(GTK_ENTRY(ibottom_margin_text), buf);
			break;

		case BCARDS_LEFTMARGIN:
			new_margin = imargin_left * 72.0 + (0.5*offsetx);
			imargin_left = new_margin/72.0;

			if ( imargin_left > (sfactor*grid_width/72)/2 )
			{
				imargin_left = (sfactor*grid_width/72)/2;
			}
			if ( imargin_left < 0 )
			{
				imargin_left = 0;
			}
			sprintf(buf, "%.3f", imargin_left);
			gtk_entry_set_text(GTK_ENTRY(ileft_margin_text), buf);
			break;

		case BCARDS_RIGHTMARGIN:
			new_margin = imargin_right * 72.0 - (0.5*offsetx);
			imargin_right = new_margin/72.0;

			if ( imargin_right > (sfactor*grid_width/72)/2 )
			{
				imargin_right = (sfactor*grid_width/72)/2;
			}
			if ( imargin_right < 0 )
			{
				imargin_right = 0;
			}
			sprintf(buf, "%.3f", imargin_right);
			gtk_entry_set_text(GTK_ENTRY(iright_margin_text), buf);
			break;
	}

	/*
	 * Redraw the screen.
	 */
	PreviewUpdate();

	/*
	 * Update Margin windows.
	 */
	sprintf(buf, "%.3f", imargin_top);
	gtk_entry_set_text(GTK_ENTRY(itop_margin_text), buf);
	sprintf(buf, "%.3f", imargin_bottom);
	gtk_entry_set_text(GTK_ENTRY(ibottom_margin_text), buf);
	sprintf(buf, "%.3f", imargin_left);
	gtk_entry_set_text(GTK_ENTRY(ileft_margin_text), buf);
	sprintf(buf, "%.3f", imargin_right);
	gtk_entry_set_text(GTK_ENTRY(iright_margin_text), buf);

}

/*========================================================================
 *	Name:			BCardsTextEntryUpdates
 *	Prototype:	void BCardsTextEntryUpdates()
 *					
 *	Description:
 *		Update the display based on input from the user.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
BCardsTextEntryUpdates(
	GtkWidget	*widget,
	gpointer		data
)
{
	char					buf[32];
	static gboolean	update=FALSE;
	gfloat				grid_width, grid_height;
	gfloat				sfactor;
	int					dpi;

	/*
	 * Determine current grid size.
	 */
	dpi = atoi(gtk_entry_get_text(GTK_ENTRY(dpi_text)));
	grid_width = (page_width - margin_left - margin_right)/columns * dpi;
	grid_height = (page_height - margin_top - margin_bottom)/rows * dpi;
	if ( grid_width > grid_height )
		sfactor = (gfloat)(IMAGE_PREVIEW_WSIZE-1) / (gfloat)grid_width;
	else
		sfactor = (gfloat)(IMAGE_PREVIEW_HSIZE-1) / (gfloat)grid_height;

	switch((int)data)
	{
		case BCARDS_PAGE_TOP_MARGIN_FIELD:
			update = FALSE;
			imargin_top = atof(gtk_entry_get_text(GTK_ENTRY(itop_margin_text)));

			if ( imargin_top > (sfactor*grid_height/72)/2 )
			{
				imargin_top = (sfactor*grid_height/72)/2;
				update = TRUE;
			}
			if ( imargin_top < 0 )
			{
				imargin_top = 0;
				update = TRUE;
			}

			if ( update )
			{
				gtk_signal_handler_block_by_func(
					GTK_OBJECT(itop_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_TOP_MARGIN_FIELD);
				sprintf(buf, "%.3f", imargin_top);
				gtk_entry_set_text(GTK_ENTRY(itop_margin_text), buf);
				gtk_signal_handler_unblock_by_func(
					GTK_OBJECT(itop_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_TOP_MARGIN_FIELD);
			}
			break;

		case BCARDS_PAGE_BOTTOM_MARGIN_FIELD:
			update = FALSE;
			imargin_bottom = 
				atof(gtk_entry_get_text(GTK_ENTRY(ibottom_margin_text)));

			if ( imargin_bottom > (sfactor*grid_height/72)/2 )
			{
				imargin_bottom = (sfactor*grid_height/72)/2;
				update = TRUE;
			}
			if ( imargin_bottom < 0 )
			{
				imargin_bottom = 0;
				update = TRUE;
			}

			if ( update )
			{
				gtk_signal_handler_block_by_func(
					GTK_OBJECT(ibottom_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_BOTTOM_MARGIN_FIELD);
				sprintf(buf, "%.3f", imargin_bottom);
				gtk_entry_set_text(GTK_ENTRY(ibottom_margin_text), buf);
				gtk_signal_handler_unblock_by_func(
					GTK_OBJECT(ibottom_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_BOTTOM_MARGIN_FIELD);
			}
			break;

		case BCARDS_PAGE_LEFT_MARGIN_FIELD:
			update = FALSE;
			imargin_left = 
				atof(gtk_entry_get_text(GTK_ENTRY(ileft_margin_text)));

			if ( imargin_left > (sfactor*grid_width/72)/2 )
			{
				imargin_left = (sfactor*grid_width/72)/2;
				update = TRUE;
			}
			if ( imargin_left < 0 )
			{
				imargin_left = 0;
				update = TRUE;
			}

			if ( update )
			{
				gtk_signal_handler_block_by_func(
					GTK_OBJECT(ileft_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_LEFT_MARGIN_FIELD);
				sprintf(buf, "%.3f", imargin_left);
				gtk_entry_set_text(GTK_ENTRY(ileft_margin_text), buf);
				gtk_signal_handler_unblock_by_func(
					GTK_OBJECT(ileft_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_LEFT_MARGIN_FIELD);
			}
			break;

		case BCARDS_PAGE_RIGHT_MARGIN_FIELD:
			update = FALSE;
			imargin_right = 
				atof(gtk_entry_get_text(GTK_ENTRY(iright_margin_text)));

			if ( imargin_right > (sfactor*grid_width/72)/2 )
			{
				imargin_right = (sfactor*grid_width/72)/2;
				update = TRUE;
			}
			if ( imargin_right < 0 )
			{
				imargin_right = 0;
				update = TRUE;
			}

			if ( update )
			{
				gtk_signal_handler_block_by_func(
					GTK_OBJECT(iright_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_RIGHT_MARGIN_FIELD);
				sprintf(buf, "%.3f", imargin_right);
				gtk_entry_set_text(GTK_ENTRY(iright_margin_text), buf);
				gtk_signal_handler_unblock_by_func(
					GTK_OBJECT(iright_margin_text), 
					(GtkSignalFunc) BCardsTextEntryUpdates, 
					(gpointer)BCARDS_PAGE_RIGHT_MARGIN_FIELD);
			}
			break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			BCardsResetFields
 *	Prototype:	void BCardsResetFields()
 *					
 *	Description:
 *		Reset the dialog to its initial configuration.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
void
BCardsResetFields()
{
	char	buf[32];
	
	sprintf(buf, "%.3f", IMARGIN_TOP_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(itop_margin_text), buf);
	sprintf(buf, "%.3f", IMARGIN_BOTTOM_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(ibottom_margin_text), buf);
	sprintf(buf, "%.3f", IMARGIN_LEFT_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(ileft_margin_text), buf);
	sprintf(buf, "%.3f", IMARGIN_RIGHT_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(iright_margin_text), buf);

	active_margin = BCARDS_TOPMARGIN;
}


/*========================================================================
 *	Name:			MarginToggle
 *	Prototype:	static void MarginToggle(GtkWidget *widget, gpointer data)
 *					
 *	Description:
 *		Set the active margin line based on which toggle has been selected.
 *
 *	Input Arguments:
 * GtkWidget	*widget		unused
 * gpointer		data			one of
 *									BCARDS_TOPMARGIN
 *									BCARDS_BOTTOMMARGIN
 *									BCARDS_LEFTMARGIN
 *									BCARDS_RIGHTMARGIN
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
MarginToggle(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch ((int)data)
	{
		case BCARDS_TOPMARGIN:    active_margin = BCARDS_TOPMARGIN; break;
		case BCARDS_BOTTOMMARGIN: active_margin = BCARDS_BOTTOMMARGIN; break;
		case BCARDS_LEFTMARGIN:   active_margin = BCARDS_LEFTMARGIN; break;
		case BCARDS_RIGHTMARGIN:  active_margin = BCARDS_RIGHTMARGIN; break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			BorderToggle
 *	Prototype:	static void BorderToggle(GtkWidget *widget, gpointer data)
 *					
 *	Description:
 *		Enable/Disable border around source image for each copy.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
BorderToggle(
	GtkWidget	*widget,
	gpointer		data
)
{
	if ( border_state == ON )
		border_state = OFF;
	else
		border_state = ON;
}

