/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1995,1996,1997,1998.
 *========================================================================
 *              FILE NAME: dialog.c
 *            DESCRIPTION: Generic Dialog handler routines
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: GFXMsgWindow, GFXMsgWindowUpdate
 *      PRIVATE FUNCTIONS: CloseMsgWindow
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define GFX_DIALOG_C

/* === System Headers === */
#include <X11/Xlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>

#include <gtk/gtk.h>


/* === Project Headers === */
#include "gfxcommon.h"
#include "info_xpm.xpm"
#include "error_xpm.xpm"
#include "msg_xpm.xpm"
#include "debug.h"


/* === external routines === */
extern GtkWidget	*GFXCreateGenericShell();
extern GtkWidget	*GFXCenterWindow();


/* === Public routine prototypes === */
void GFXMsgWindowUpdate();


/* === Private routine prototypes === */
static void CloseMsgWindow();


/* === Global Variables === */


/* === Static Variables === */
static GtkWidget	*messagebox=NULL;
static GdkPixmap	*error_xbm;
static GdkPixmap	*info_xbm;
static GdkPixmap	*msg_xbm;
static GdkBitmap	*msg_mask, *info_mask, *error_mask;



/*========================================================================
 *	Name:			GFXMsgWindow
 *	Prototype:	GFXMsgWindow(char *message, int type, 
 *							GtkWidget **return_widget, GtkWidget **data_widget,
 *							GtkWidget *parent,
 *							gpointer ok_cb, gpointer ok_data,
 *							gpointer cancel_cb, gpointer help_cb,
 *							int req_width, int req_height)
 *
 *	Description:
 *		This routine creates and pops up a generic message window.
 *		The content of the message is passed from the caller and the
 *		type of icon to use with the message is also defined by the
 *		caller.
 *
 *	Input Arguments:
 *		char 			*message		Message to display
 *		int 			type			Defines which icon to use in window
 *										Valid values are:
 *										GFX_ERROR_TYPE    - Error message
 *										GFX_MSG_TYPE	   - Generic message
 *										GFX_INFO_TYPE     - Informational message
 *										GFX_CONFIRM_TYPE  - Informational message
 *										GFX_MSG_NOCANCEL  - Don't use cancel button
 *										GFX_MSG_NOHELP    - Don't use Help button
 *										GFX_MSG_NOOK      - Don't use OK button
 *		GtkWidget	*parent		Parent widget, in the center of which the message
 *										window will be opened.
 *		gpointer		ok_cb			callback to use for OK button
 *										defaults to CloseMsgWindow() if not specified.
 *		gpointer		ok_data		user data if OK callback is specified
 *		gpointer		cancel_cb	callback to use for Close button
 *										defaults to CloseMsgWindow() if not specified.
 *		gpointer		help_cb		callback to use for Help button
 *										no default callback for this is provided.
 *		int			req_width	requested width for text widget; 0 means to
 *										use the default GFX_DIALOG_WIDTH
 *		int			req_height	requested height for text widget; 0 means to
 *										use the default GFX_DIALOG_HEIGHT
 *
 *	Output Arguments:
 *		GtkWidget	**return_widget	if non-NULL, return dialog widget 
 *												to caller here
 *		GtkWidget	**data_widget		if non-NULL, return text widget
 *												to caller here
 *
 *	Return Values:
 *		None
 *
 *	Global Variables:
 *	External Routines:
 *	Method:
 *	Restrictions:
 *	Notes:
 * This routine does allow multiple dialogs to be opened.  If you 
 * call this routine you should save the dialog widget id and close
 * the dialog yourself.
 *
 * If you don't specify one or more of GFX_MSG_NOCANCEL, GFX_MSG_NOHELP
 * or GFX_MSG_NOOK then you get three buttons in the dialog: Accept,
 * Cancel, and Help.  
 *
 * GFX_MSG_NOOK removes the Accept button.
 * GFX_MSG_NOCANCEL removes the Cancel button.
 * GFX_MSG_NOHELP removes the Help button.
 *
 * The types determine what title is used for the dialog and which 
 * pixmap to use on the left side of the dialog.
 * GFX_MSG_TYPE and GFX_CONFIRM_TYPE get the msg_xpm.xpm pixmap
 * GFX_INFO_TYPE gets the info_xpm.xpm pixmap
 * GFX_ERROR_TYPE gets the error_xpm.xpm pixmap
 *	
 *========================================================================*/
void
GFXMsgWindow(
	char			*message,			/* text message to display */
	int			type,					/* type of dialog to use */
	GtkWidget	**return_widget,	/* where to save dialog widget id */
	GtkWidget	**data_widget,		/* where to save text widget id */
	GtkWidget	*parent,				/* who the parent of this dialog will be */
	gpointer		ok_cb,				/* Callback for OK/ACCEPT button */
	gpointer		ok_data,				/* User data for OK/ACCEPT button */
	gpointer		cancel_cb,			/* Callback for CANCEL button */
	gpointer		help_cb,				/* Callback for HELP button */
	int			req_width,			/* width of text widget; 0 = use default */
	int			req_height			/* height of text widget; 0 = use default */
)
{
#ifdef DEBUG
	char			fname[]="GFXMsgWindow()";
#endif

	char				title[256];
	GtkWidget		*vbox, *hbox, *accept_button, *close_button, *help_button;
	GtkWidget		*vscrollbar;
	GtkWidget		*label_pixmap;
	GtkWidget		*data_text;
	GdkPixmap		*current_xbm=NULL;
	GdkBitmap		*current_mask=NULL;
	GtkStyle			*style=NULL;
	int				shell_type;
	GShell_T			gshell;
	int				width, height;

	static int		firsttime = TRUE;


	DBGEnter();

	/*
	 * Create the dialog. We'll destroy it when it is closed.
	 */
	if (type & GFX_ERROR_TYPE)
		(void) sprintf ( title, "%s", GFX_S_ERROR_TITLE );
	else
	if (type & GFX_MSG_TYPE)
		(void) sprintf ( title, "%s", GFX_S_MSG_TITLE );
	else
	if (type & GFX_INFO_TYPE)
		(void) sprintf ( title, "%s", GFX_S_INFO_TITLE );
	else
		(void) sprintf ( title, "%s", GFX_S_CONFIRM_TITLE );

	if ( type & GFX_MSG_NOCANCEL & GFX_MSG_NOHELP & GFX_MSG_NOOK )
	{
		shell_type = 0;
	}
	else
	{
		if ( type & GFX_CONFIRM_TYPE )
		{
			shell_type =  YES_BUTTON | NO_BUTTON | HELP_BUTTON;

			if ( type & GFX_MSG_NOCANCEL )
				shell_type = shell_type ^ NO_BUTTON;
	
			if ( type & GFX_MSG_NOHELP )
				shell_type = shell_type ^ HELP_BUTTON;
	
			if ( type & GFX_MSG_NOOK )
				shell_type = shell_type ^ YES_BUTTON;
		}
		else
		{
			shell_type =  ACCEPT_BUTTON | CLOSE_BUTTON | HELP_BUTTON;

			if ( type & GFX_MSG_NOCANCEL )
				shell_type = shell_type ^ CLOSE_BUTTON;
	
			if ( type & GFX_MSG_NOHELP )
				shell_type = shell_type ^ HELP_BUTTON;
	
			if ( type & GFX_MSG_NOOK )
				shell_type = shell_type ^ ACCEPT_BUTTON;
		}
	}

	messagebox = GFXCreateGenericShell(title, shell_type, &gshell, TRUE );

	gtk_signal_connect (GTK_OBJECT (messagebox), "destroy",
			GTK_SIGNAL_FUNC (CloseMsgWindow), messagebox);

	vbox = gshell.vbox;
	close_button = gshell.cancel;
	accept_button = gshell.accept;
	help_button = gshell.help;

	DBGPrintf(DBG_INFO,("vbox: %d\n", (int)vbox));

	/*
	 * We want a horizontal display, so create a hbox in the vbox.
	 */
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_widget_show(hbox);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	/*
	 * First time through create the pixmaps - we need to do this after the
	 * label has been created so we can use it as the color to map the
	 * background colors to.
	 */
	if ( firsttime )
	{
		DBGPrintf(DBG_INFO,("Creating dialog pixmaps\n"));

		style = gtk_widget_get_style( messagebox );

		msg_xbm = gdk_pixmap_create_from_xpm_d(
			messagebox->window,
			&msg_mask,
			&style->bg[GTK_STATE_NORMAL],
			msg_xpm);
		DBGPrintf(DBG_INFO,("MSG pixmap: %d\n", (int) msg_xbm));

		info_xbm = gdk_pixmap_create_from_xpm_d(
			messagebox->window,
			&info_mask,
			&style->bg[GTK_STATE_NORMAL],
			info_xpm);
		DBGPrintf(DBG_INFO,("INFO pixmap: %d\n", (int) info_xbm));

		error_xbm = gdk_pixmap_create_from_xpm_d(
			messagebox->window,
			&error_mask,
			&style->bg[GTK_STATE_NORMAL],
			error_xpm);
		DBGPrintf(DBG_INFO,("ERROR pixmap: %d\n", (int) error_xbm));

		firsttime = FALSE;
	}

	/*
	 * Setup the buttons in the window based on the type of message
	 * window requested.
	 */
	if (( type & GFX_MSG_TYPE ) || ( type & GFX_CONFIRM_TYPE ))
	{
		current_xbm = msg_xbm;
		current_mask = msg_mask;
		DBGPrintf(DBG_INFO,("using MSG pixmap: %d\n", (int) current_xbm));
	}

	if ( type & GFX_INFO_TYPE )
	{
		DBGPrintf(DBG_INFO,("using INFO pixmap\n"));
		current_xbm = info_xbm;
		current_mask = info_mask;
	}

	if ( type & GFX_ERROR_TYPE )
	{
		DBGPrintf(DBG_INFO,("using ERROR pixmap\n"));
		current_xbm = error_xbm;
		current_mask = error_mask;
	}

	/*
	 * Set the message boxes pixmap.
	 */
	DBGPrintf(DBG_INFO,("pixmap: %d\n", (int) current_xbm));
	label_pixmap = gtk_pixmap_new(current_xbm, current_mask);
	gtk_widget_show(label_pixmap);
	gtk_box_pack_start(GTK_BOX(hbox), label_pixmap, FALSE, FALSE, 0);

	data_text = gtk_text_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), data_text, TRUE, TRUE, 0);
	gtk_text_set_editable (GTK_TEXT (data_text), FALSE);
	gtk_text_set_word_wrap(GTK_TEXT(data_text), TRUE);
	if ( req_width == 0 )
		width = GFX_DIALOG_WIDTH;
	else
		width = req_width;
	if ( req_height == 0 )
		height = GFX_DIALOG_HEIGHT;
	else
		height = req_height;
	gtk_widget_set_usize(data_text, width, height);
	gtk_widget_realize(data_text);
	if ( message != NULL )
	{
		gtk_text_freeze(GTK_TEXT(data_text));
		gtk_text_insert (
			GTK_TEXT (data_text),      /* widget */
			NULL,                      /* font */
			&data_text->style->black,  /* fg color */
			NULL,                      /* bg color */
			message,                   /* text to insert */
			-1);                       /* use strlen() for text size */
		gtk_text_thaw (GTK_TEXT (data_text));
		gtk_text_set_point(GTK_TEXT(data_text), strlen(message));
	}
	gtk_widget_show(data_text);

	vscrollbar = gtk_vscrollbar_new (GTK_TEXT(data_text)->vadj);
	gtk_box_pack_start(GTK_BOX(hbox), vscrollbar, FALSE, FALSE, 0);
	gtk_widget_show (vscrollbar);
	gtk_widget_set_usize(vscrollbar, 11, -1);

	/*
	 * Set the callbacks for the buttons.
	 */
	if ( ! (type & GFX_MSG_NOCANCEL) )
	{
		if ( cancel_cb != NULL )
			gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
				GTK_SIGNAL_FUNC (cancel_cb), messagebox);
		else
			gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
				GTK_SIGNAL_FUNC (CloseMsgWindow), messagebox);
	}
	if ( ! (type & GFX_MSG_NOOK) )
	{
		if ( ok_cb != NULL )
			gtk_signal_connect (GTK_OBJECT (accept_button), "clicked",
				GTK_SIGNAL_FUNC (ok_cb), ok_data);
		else
			gtk_signal_connect (GTK_OBJECT (accept_button), "clicked",
				GTK_SIGNAL_FUNC (CloseMsgWindow), messagebox);
	}
	if ( ! (type & GFX_MSG_NOHELP) )
	{
		if ( help_cb != NULL )
			gtk_signal_connect (GTK_OBJECT (help_button), "clicked",
				GTK_SIGNAL_FUNC (help_cb), NULL);
	}

	gtk_widget_show(messagebox);

	/*
	 * Center the dialog in the display.
	 */
	GFXCenterWindow(messagebox);


	/*
	 * Pass back the messagebox widgets id, if requested, so the caller
	 * can pop down the dialog.
	 */
	if ( return_widget != NULL )
		*return_widget = messagebox;
	if ( data_widget != NULL )
		*data_widget = data_text;
	
	/*
	 * Try to force an update assuming we have some expose events
	 * generated from the building of this dialog.
	 */
	gtk_main_iteration();
	
	DBGExit();
}


/*========================================================================
 *	Name:			CloseMsgWindow
 *	Prototype:	CloseMsgWindow()
 *
 *	Description:
 *		Default callback for the OK and Close buttons that closes the
 *		dialog window if no other callback has been specified.  If the
 *		user provides callbacks, then they also have to handle closing
 *		the dialog window.
 *
 *	Notes:
 *		
 *========================================================================*/
static void
CloseMsgWindow(
	GtkWidget	*widget,
	gpointer		shell
)
{
	if ( GTK_WIDGET(shell) != NULL )
		gtk_widget_destroy(GTK_WIDGET(shell));
}



/*========================================================================
 *	Name:			GFXMsgWindowUpdate
 *	Prototype:	GFXMsgWindowUpdate()
 *
 *	Description:
 *		Update the message window for an existing dialog.
 *
 *	Notes:
 *		
 *========================================================================*/
void
GFXMsgWindowUpdate(
	char			*message,
	GtkWidget	*text_widget
)
{
	int				length;

	if ( ( message != NULL ) && ( text_widget != NULL ) )
	{
		gtk_text_freeze(GTK_TEXT(text_widget));
		gtk_text_insert (
			GTK_TEXT (text_widget),    /* widget */
			NULL,                      /* font */
			&text_widget->style->black,/* fg color */
			NULL,                      /* bg color */
			message,                   /* text to insert */
			-1);                       /* use strlen() for text size */
		gtk_text_thaw (GTK_TEXT (text_widget));

		length = gtk_text_get_length(GTK_TEXT(text_widget));
		gtk_text_set_point(GTK_TEXT(text_widget), length);
	}

	/*
	 * Attempt to force scrolled window to bottom of window after
	 * each addition to make sure we get the latest data displayed.
	adjustment = GTK_TEXT(text_widget)->vadj;
	gtk_signal_emit_by_name (GTK_OBJECT (adjustment), "changed");
	 */

	/*
	 * Try to force an update assuming we have some expose events
	 * generated from the building of this dialog.
	 */
	gtk_main_iteration();
}


