/* $Id$ */

/*========================================================================
 *  Copyright (c) Michael J. Hammel 1998.
 *========================================================================
 *              FILE NAME: lightning.c
 *            DESCRIPTION: main module for BoltGFX plug-in
 *      DEFINED CONSTANTS: 
 *       TYPE DEFINITIONS: 
 *      MACRO DEFINITIONS: 
 *       GLOBAL VARIABLES: 
 *       PUBLIC FUNCTIONS: 
 *      PRIVATE FUNCTIONS: 
 *  SOFTWARE DEPENDENCIES: 
 *  HARDWARE DEPENDENCIES: 
 *                  NOTES: 
 *
 * SPECIAL CONSIDERATIONS:
 * Set your tabstops to 3 to make the code more readable.
 *========================================================================
 *
 * MODIFICATION HISTORY:
 * $Log$
 *
 *========================================================================*/
#define MAIN_C		/* needed by debug library */

/* === System Headers === */
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <math.h>

/*
 * Assumes we have the GTK and gimp.h installed in places that the compiler
 * knows how to find or that we've specified where to find them using the
 * Makefile.
 */
#include <gtk/gtk.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>


/* === Project Headers === */
#include "lightning.h"
#include "gfxcommon.h"
#include "gm-logo.xpm"
#include "debug.h"


/*
 * _GimpPlugInInfo is a structure that tells the gimp what routines to
 * run at certain times.  For most plug-ins, init_proc() and 
 * quit_proc() are unnecessary.  All plug-ins will want to define
 * a query_proc() and a run_proc().
 */
typedef struct _GimpPlugInInfo GPlugInfo;


/* === external routines === */
extern void GFXMsgWindow();
extern void GFXMsgWindowUpdate();
extern void GFXCenterWindow();


/* === Public routine prototypes === */
static void BoltGFXQuery();	/* static, but made global with PLUG_IN_INFO */
static void BoltGFXRun();		/* static, but made global with PLUG_IN_INFO */


/* === Private routine prototypes === */
static void BoltGFX();		/* the heart of the plug-in */
static void CloseCallback();
static void OKCallback();
static void PreviewUpdate();
static void PreviewButtonPress();
static void PreviewButtonRelease();
static void PreviewMotion();
static void ResetFields();
static void TypeToggle();
static gint LayerConstraints();
static void LayerSelect();
static void ArrowCallback();
static void TextEntryUpdates();
static void ShowStatus();
static void ShowHelp();
static void BlendSelect();
static void OpacityTextUpdate();
static void OpacitySliderUpdate();
static void RebuildLayerMenu();


/* === Global Variables === */
GimpPlugInInfo PLUG_IN_INFO = {	/* Required:  you *must* name this PLUG_IN_INFO */
	NULL,				/* No Init procedure */
	NULL,				/* No Exit procedure */
	BoltGFXQuery,	/* Query procedure */
	BoltGFXRun,	/* Run time procedure - where the work really starts */
};


GtkWidget		*boltgfx_dialog;
GtkWidget		*new_layer_radio_button;
GtkWidget		*smooth_button;
GtkWidget		*angle_text;
GtkDrawingArea	*preview;
GtkWidget		*msg_widget;
GtkWidget		*msg_data;
GtkWidget		*status_label;
GtkWidget		*help_widget;
GtkWidget		*blend_menu;
GtkWidget		*layer_options;
GtkWidget		*layer_menu;
GtkWidget		*opacity_text;
GtkWidget		*opacity_slider;
GtkObject		*opacity_adj;

GimpDrawable		*drawable;
gint32			drawable_id;

GdkPoint	polycorners[4];		/* XY coords for corners of page in preview */
int		active_type;			/* new layer or anchor to original? */
int		grab_box_active=FALSE;	/* active if user clicks in a grab box */
gfloat	rotation_angle=0;		/* current rotation amount */
int		grabbox_x, grabbox_y;/* hot spot of page rotation grab box */
int		mouse_x, mouse_y;		/* current location of mouse on mouse clicks */
int		page_rotate;			/* if True: mouse drags rotate page */

int		layer_id;				/* ID of layer to use as source */
int		image_id;				/* ID of image initial layer lives in */
gfloat	keymod = 1.0;			/* modifier for rotation calculations */
int		blend_mode = 0;		/* NORMAL mode by default for new layers */
int		keypress;


static char *blend_strings[] = {
	"Normal",
	"Dissolve",
	"Behind",
	"Multiply",
	"Screen",
	"Overlay",
	"Difference",
	"Addition",
	"Subtract",
	"Darken Only",
	"Lighten Only",
	"Hue",
	"Saturation",
	"Color",
	"Value",
	NULL
};


/*========================================================================
 *	Name:			BoltGFXQuery
 *	Prototype:	static void BoltGFXQuery()
 *					
 *	Description:
 *		Register this plug-in in the Gimp Procedural Database.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *	Most of the gimp_install_procedure() arguments are defined values that
 * can be found in the lightning.h file.
 *========================================================================*/
static void
BoltGFXQuery()
{
	static GimpParamDef args[] = {
		{ GIMP_PDB_INT32,		"run_mode",	"Interactive, non-interactive" },
		{ GIMP_PDB_IMAGE,		"image",		"Input image" },
		{ GIMP_PDB_DRAWABLE,	"drawable",	"Input drawable" },
		{ GIMP_PDB_LAYER,		"layer",		"Input layer" },
		{ GIMP_PDB_INT32,		"smooth",	"If TRUE, smooths jaggies after rotation" },
		{ GIMP_PDB_INT32,		"type",		"0: Create new layer 1: Anchor to original" },
	};
	static int	nargs = sizeof(args) / sizeof(args[0]);

	gimp_install_procedure(
		TRANSGFX_NAME,
		TRANSGFX_BLURB,
		TRANSGFX_HELP,
		TRANSGFX_AUTHOR,
		TRANSGFX_COPYRIGHT,
		TRANSGFX_DATE,
		TRANSGFX_MENU_PATH,
		TRANSGFX_IMAGE_TYPES,
		TRANSGFX_PROC_TYPE,
		nargs,
		0,
		args,
		NULL
	);
}


/*========================================================================
 *	Name:			BoltGFXRun
 *	Prototype:	static void BoltGFXRun()
 *					
 *	Description:
 *		Starts the plug-in.  Basically just preps for creatingg the dialog
 *		and does some initialization before calling the heart of the 
 *		plugin, BoltGFX().
 *
 *	Input Arguments:
 *	char		*name				Name of print program.
 *	int		nparams			Number of parameters passed in
 *	GimpParam	*param			Parameter values
 *
 *	Output Arguments:
 *	int		*nreturn_vals	Number of return values
 *	GimpParam	**return_vals	Return values
 *
 *	Return Values:
 *	Method:
 *	Restrictions:
 *	Notes:
 *========================================================================*/
static void
BoltGFXRun(
	char		*name,			/* Name of print program. */
	int		nparams,			/* Number of parameters passed in */
	GimpParam	*param,			/* Parameter values */
	int		*nreturn_vals,	/* Number of return values */
	GimpParam	**return_vals	/* Return values */
)
{
	GimpRunModeType	run_mode;
	GimpParam			*values;

	/*
	 * Initialize parameter data...
	 */
	run_mode = param[0].data.d_int32;
	values = g_new(GimpParam, 1);
	values[0].type          = GIMP_PDB_STATUS;
	values[0].data.d_status = GIMP_PDB_SUCCESS;
	*nreturn_vals = 1;
	*return_vals  = values;
 
	/*
	 * Get drawable.  This is what we will be scaling and duplicating
	 * unless the user selects another one.
	 */
	drawable = gimp_drawable_get(param[2].data.d_drawable);
	drawable_id = param[2].data.d_drawable;

	image_id = param[1].data.d_layer;
	layer_id = param[3].data.d_layer;


	/*
	 * Do run-mode specific setup.
	 */
	switch(run_mode)
	{
		case GIMP_RUN_INTERACTIVE:
			/*
			gimp_get_data(TRANSGFX_PI_NAME, &init_values);
			*/

			BoltGFX();
			break;

		case GIMP_RUN_NONINTERACTIVE:
			break;

		case GIMP_RUN_WITH_LAST_VALS:
			break;

		default:
			values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
			break;
	}
}



/*========================================================================
 *	Name:			MAIN
 *	Prototype:	MAIN()
 *					
 *	Description:
 *		Gimp macro that defines a plug-in's main routine.  All plug-ins must
 *		include this.
 *========================================================================*/
MAIN();



/*========================================================================
 *	Name:			BoltGFX
 *	Prototype:	static void BoltGFX()
 *					
 *	Description:
 *		Set up the dialog and display it.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *		Get the requested drawable and save a copy of it.
 *		Prompt user for input options (re: the dialog window).
 *		If a selection is present, float selection.
 *		If anchoring has been requested, anchor floating selection back to
 *			original layer.
 *		else
 *			set new layer blend mode.
 *
 *	Restrictions:
 *========================================================================*/
static void
BoltGFX()
{
	gint			argc;
	gchar			**argv;
	char			buf[256];
	int			index;

	GtkWidget	*table, *toggle_table;
	GtkWidget	*vbox, *hbox, *frame, *hsep;
	GtkWidget	*label;
	GtkWidget	*label_pixmap;
	GtkWidget	*arrow;
	GtkWidget	*option_menu, *menu, *menuitem;
	GtkWidget	*toggle;
	GtkWidget	*reset_button;
	GtkWidget	*ok_button, *cancel_button, *help_button, *status_button;
	GtkWidget	*angle_up_button;
	GtkWidget	*angle_down_button;
	GdkPixmap	*logo;
	GdkBitmap	*logo_mask;
	GtkStyle		*style;

	GSList		*group = NULL;

	/*
	 * Generic initialization:
	 * 0. Fake the argc/argv stuff.
	 * 1. Initialize the GTK toolkit.
	 * 2. parse the Gimp's gtkrc for plug-in specific settings.
	 * 3. Set up to use XSharedMemory, if possible.
	 */
	argc    = 1;
	argv    = g_new(gchar *, 1);
	argv[0] = g_strdup("bcards");
	gtk_init(&argc, &argv);
	gtk_rc_parse(gimp_gtkrc());
	gdk_set_use_xshm(gimp_use_xshm());

	/*
	 * Try to behave in a civil manner should the unexpected occur.
	 */
	signal(SIGBUS, SIG_DFL);
	signal(SIGSEGV, SIG_DFL);

	/*
	 * Create a new dialog window.
	 */
	boltgfx_dialog = gtk_dialog_new();
	sprintf(buf, "%s - %s", TRANSGFX_TITLE, TRANSGFX_VERSION);
	gtk_window_set_title(GTK_WINDOW(boltgfx_dialog), buf);
	gtk_window_set_wmclass(GTK_WINDOW(boltgfx_dialog), "boltgfx", "Gimp");
	gtk_window_position(GTK_WINDOW(boltgfx_dialog), GTK_WIN_POS_MOUSE);
	gtk_container_border_width(GTK_CONTAINER(boltgfx_dialog), 0);
	gtk_signal_connect(GTK_OBJECT(boltgfx_dialog), "destroy",
		(GtkSignalFunc)CloseCallback, NULL);

	gtk_widget_realize(boltgfx_dialog);

	/*
	 * Initialize the logo.  
	 */
	style = gtk_widget_get_style( boltgfx_dialog );
	logo = gdk_pixmap_create_from_xpm_d(
					boltgfx_dialog->window, 
					&logo_mask, 
					&style->bg[GTK_STATE_NORMAL], 
					gm_logo_xpm);

	/*
	 * The main table in which everything else will go.
	 */
	table = gtk_table_new(5, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 1);
	gtk_table_set_col_spacings(GTK_TABLE(table), 1);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (boltgfx_dialog)->action_area), 
		table, TRUE, TRUE, 0);
	gtk_widget_show(table);

	/*
	 * A small preview window that will be used for rotating interactively.
	 */
	frame = gtk_frame_new("Rotation Angle Preview");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);

	preview = (GtkDrawingArea *)gtk_drawing_area_new();
	gtk_drawing_area_size(preview, PREVIEW_WSIZE, PREVIEW_HSIZE);
	gtk_container_add (GTK_CONTAINER (frame), (GtkWidget *)preview);
	gtk_widget_show((GtkWidget *)preview);

	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "expose_event",
			(GtkSignalFunc)PreviewUpdate,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_press_event",
			(GtkSignalFunc)PreviewButtonPress,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "button_release_event",
			(GtkSignalFunc)PreviewButtonRelease,
			NULL);
	gtk_signal_connect(GTK_OBJECT((GtkWidget *)preview), "motion_notify_event",
			(GtkSignalFunc)PreviewMotion,
			NULL);

	gtk_widget_set_events((GtkWidget *)preview,
		GDK_EXPOSURE_MASK | GDK_BUTTON_MOTION_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
 

	/*
	 * A smaller table in which we pack some toggle and radio buttons.
	 */
	toggle_table = gtk_table_new(5, 1, FALSE);
	gtk_container_border_width(GTK_CONTAINER(toggle_table), 6);
	gtk_table_set_col_spacings(GTK_TABLE(toggle_table), 4);
	gtk_table_set_row_spacings(GTK_TABLE(toggle_table), 8);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)toggle_table, 
		1, 2, 0, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(toggle_table);


	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(toggle_table), (GtkWidget *)frame, 
		0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show(frame);
	label_pixmap = gtk_pixmap_new(logo, logo_mask);
	gtk_container_add (GTK_CONTAINER (frame), label_pixmap);
	gtk_widget_show(label_pixmap);


	/*
	 * Radio buttons for New Layer and Anchor options.
	 */
	frame = gtk_frame_new("Disposition Of Rotation");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(toggle_table), (GtkWidget *)frame, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 2);
	gtk_container_add (GTK_CONTAINER (frame), vbox);
	gtk_widget_show(vbox);

	new_layer_radio_button = toggle = 
		gtk_radio_button_new_with_label(NULL, "New Layer");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, TRUE, TRUE, 4);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle), TRUE);
	active_type = TRANSGFX_TYPE_NEW_LAYER;
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) TypeToggle, (gpointer) TRANSGFX_TYPE_NEW_LAYER);
	gtk_widget_show (toggle);

	toggle = gtk_radio_button_new_with_label(group, "Anchor to original");
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (toggle));
	gtk_box_pack_start (GTK_BOX (vbox), toggle, TRUE, TRUE, 4);
	gtk_signal_connect (GTK_OBJECT (toggle), "toggled",
		(GtkSignalFunc) TypeToggle, (gpointer) TRANSGFX_TYPE_ANCHOR);
	gtk_widget_show (toggle);

	/*
	 * Smooth button.
	 */
	smooth_button = gtk_check_button_new_with_label("Smooth Jagged Edges");
	gtk_table_attach(GTK_TABLE(toggle_table), (GtkWidget *)smooth_button, 
		0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show(smooth_button);


	/*
	 * Layer Blend menu and Opacity slider.
	 * Sets default blend mode for new layers.
	 */
	hbox = gtk_hbox_new(FALSE, 4);
	gtk_table_attach(GTK_TABLE(toggle_table), (GtkWidget *)hbox, 
		0, 1, 3, 4, 0, 0, 0, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new("Blend Mode:");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	blend_menu = option_menu = gtk_option_menu_new();
	gtk_box_pack_start (GTK_BOX (hbox), option_menu, TRUE, TRUE, 0);
	gtk_widget_show(option_menu);

	menu = gtk_menu_new();
	index=0;
   while ( blend_strings[index] != NULL )
   {
      menuitem = gtk_menu_item_new_with_label(blend_strings[index]);
      gtk_menu_append(GTK_MENU(menu), menuitem);
      gtk_widget_show(menuitem);
      gtk_signal_connect_object(GTK_OBJECT(menuitem), "activate",
         GTK_SIGNAL_FUNC(BlendSelect), (gpointer) index);
      index++;
   }  
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);


	/*
	 * Opacity slider.
	 */
	frame = gtk_frame_new("Opacity");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(toggle_table), (GtkWidget *)frame, 
		0, 1, 4, 5, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	opacity_text = gtk_entry_new_with_max_length(5);
	gtk_box_pack_start (GTK_BOX (hbox), opacity_text, FALSE, FALSE, 0);
	gtk_widget_set_usize( opacity_text, 50, -1 );
	sprintf(buf, "%.1f", OPACITY_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(opacity_text), buf);
	gtk_signal_connect(GTK_OBJECT(opacity_text), "changed",
		(GtkSignalFunc)OpacityTextUpdate, NULL);
	gtk_widget_show(opacity_text);

	opacity_adj = 
		gtk_adjustment_new(OPACITY_DEFAULT, 0.0, 110.0, 1.0, 10.0, 10.0);
	opacity_slider = gtk_hscale_new(GTK_ADJUSTMENT(opacity_adj));
	gtk_box_pack_start (GTK_BOX (hbox), opacity_slider, FALSE, FALSE, 0);
	gtk_widget_set_usize( opacity_slider, 135, -1 );
	gtk_widget_show(opacity_slider);

	gtk_signal_connect(GTK_OBJECT(opacity_adj), "value_changed",
		(GtkSignalFunc)OpacitySliderUpdate, NULL);


	/*
	 * Angle in degrees input field and arrow buttons.
	 */
	frame = gtk_frame_new("Angle (in degrees)");
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)frame, 
		0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	gtk_widget_show(hbox);

	/* Arrow Down */
	angle_down_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (angle_down_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (angle_down_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)TRANSGFX_DOWN);
	gtk_box_pack_start (GTK_BOX (hbox), angle_down_button, TRUE, TRUE, 4);
	gtk_widget_show(angle_down_button);

	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (angle_down_button), arrow);
	gtk_widget_show(arrow);

	/* Input field */
	angle_text = gtk_entry_new_with_max_length(7);
	gtk_widget_set_usize( angle_text, 70, -1 );
	gtk_box_pack_start (GTK_BOX (hbox), angle_text, TRUE, TRUE, 4);
	gtk_widget_show(angle_text);
	sprintf(buf, "%.3f", (gfloat)rotation_angle);
	gtk_entry_set_text(GTK_ENTRY(angle_text), buf);
	gtk_signal_connect (GTK_OBJECT (angle_text), "changed",
		(GtkSignalFunc) TextEntryUpdates, (gpointer)NULL);

	/* Arrow Up */
	angle_up_button = gtk_button_new();
	GTK_WIDGET_SET_FLAGS (angle_up_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (angle_up_button), "clicked",
		(GtkSignalFunc) ArrowCallback, (gpointer)TRANSGFX_UP);
	gtk_box_pack_start (GTK_BOX (hbox), angle_up_button, TRUE, TRUE, 4);
	gtk_widget_show(angle_up_button);

	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (angle_up_button), arrow);
	gtk_widget_show(arrow);


	/*
	 * Layers menu - used to select which layer to use to do rotation on.
	 */
	hbox = gtk_hbox_new(FALSE, 8);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)hbox, 
		0, 2, 2, 3, GTK_FILL , GTK_FILL , 4, 4);
	gtk_widget_show(hbox);

	label = gtk_label_new("Layer:");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	layer_menu = option_menu = gtk_option_menu_new();
	gtk_box_pack_start (GTK_BOX (hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	layer_options = menu = gimp_layer_menu_new(LayerConstraints,
					LayerSelect,
					NULL,
					0);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(option_menu);


	/*
	 * Horizontal separator between main window area and status buttons.
	 */
	hsep = gtk_hseparator_new();
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)hsep, 
		0, 2, 3, 4, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(hsep);


	/*
	 * Buttons go along the bottom row of the table.
	 */
	hbox = gtk_hbox_new(TRUE, 2);
	gtk_table_attach(GTK_TABLE(table), (GtkWidget *)hbox, 
		0, 2, 4, 5, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show(hbox);

	/*
	 * Ok Button
	 */
	ok_button = gtk_button_new_with_label ("OK");
	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
		(GtkSignalFunc) OKCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), ok_button, TRUE, TRUE, 2);
	gtk_widget_grab_default (ok_button);
	gtk_widget_show(ok_button);

	GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default (ok_button);

	/*
	 * Cancel Button
	 */
	cancel_button = gtk_button_new_with_label ("Cancel");
	GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
	gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		(GtkSignalFunc) CloseCallback, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), cancel_button, TRUE, TRUE, 2);
	gtk_widget_show(cancel_button);

	/*
	 * Reset Button
	 */
	reset_button = gtk_button_new_with_label("Reset");
	GTK_WIDGET_SET_FLAGS (reset_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(reset_button), "clicked",
		(GtkSignalFunc)ResetFields, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), reset_button, TRUE, TRUE, 2);
	gtk_widget_show(reset_button);

	/*
	 * Status Button
	 */
	status_button = gtk_button_new();

	status_label = gtk_label_new("Show Status");
	gtk_container_add (GTK_CONTAINER (status_button), (GtkWidget *)status_label);
	gtk_widget_show(status_label);

	GTK_WIDGET_SET_FLAGS (status_button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(status_button), "clicked",
		(GtkSignalFunc)ShowStatus, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), status_button, TRUE, TRUE, 2);
	gtk_widget_show(status_button);

	/*
	 * Help Button
	 */
	help_button = gtk_button_new_with_label ("Help");
	GTK_WIDGET_SET_FLAGS (help_button, GTK_CAN_DEFAULT);
	gtk_box_pack_start (GTK_BOX (hbox), help_button, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(help_button), "clicked",
		(GtkSignalFunc)ShowHelp, NULL);
	gtk_widget_show(help_button);

	/*
	 * Show the main dialog.
	 */
	gtk_widget_show(boltgfx_dialog);


	/*
	 * Sit and spin.
	 */
	gtk_main();

}


/*========================================================================
 *	Name:			ShowHelp
 *	Prototype:	static void ShowHelp()
 *					
 *	Description:
 *		Manage the Help dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowHelp(
)
{
	if ( help_widget == NULL )
	{
		GFXMsgWindow(
			GFX_HELP_TEXT,
			GFX_INFO_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&help_widget, NULL, 
			NULL, ShowHelp, NULL, ShowHelp, NULL,
			600, 400);

	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (help_widget) )
		{
			gtk_widget_hide(help_widget);
		}
		else
		{
			gtk_widget_show(help_widget);
			GFXCenterWindow(help_widget);
		}
	}
}



/*========================================================================
 *	Name:			ShowStatus
 *	Prototype:	static void ShowStatus()
 *					
 *	Description:
 *		Manage the status dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ShowStatus(
)
{
	if ( msg_widget == NULL )
	{
		GFXMsgWindow(
			"BoltGFX Status\n---------------------\n",
			GFX_MSG_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			&msg_widget, &msg_data, 
			NULL, ShowStatus, NULL, ShowStatus, NULL, 0, 0);

		gtk_label_set(GTK_LABEL(status_label), "Hide Status");
	}
	else
	{
		if ( GTK_WIDGET_VISIBLE (msg_widget) )
		{
			gtk_label_set(GTK_LABEL(status_label), "Show Status");
			gtk_widget_hide(msg_widget);
		}
		else
		{
			gtk_label_set(GTK_LABEL(status_label), "Hide Status");
			gtk_widget_show(msg_widget);
			GFXCenterWindow(msg_widget);
		}
	}
}



/*========================================================================
 *	Name:			CloseCallback
 *	Prototype:	static void CloseCallback()
 *					
 *	Description:
 *		Close the main dialog.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
CloseCallback(
)
{
	gimp_drawable_detach(drawable);
	gtk_widget_destroy(boltgfx_dialog);
	gtk_main_quit();
}


/*========================================================================
 *	Name:			TypeToggle
 *	Prototype:	static void TypeToggle(GtkWidget *widget, gpointer data)
 *					
 *	Description:
 *		Set the active type based on which toggle has been selected.
 *
 *	Input Arguments:
 * GtkWidget	*widget		unused
 * gpointer		data			one of
 *									TRANSGFX_TYPE_NEW_LAYER
 *									TRANSGFX_TYPE_ANCHOR
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
TypeToggle(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch ((int)data)
	{
		case TRANSGFX_TYPE_NEW_LAYER: 
			active_type = TRANSGFX_TYPE_NEW_LAYER; 
			gtk_widget_set_sensitive(blend_menu, TRUE);
			gtk_widget_set_sensitive(opacity_slider, TRUE);
			gtk_widget_set_sensitive(opacity_text, TRUE);
			break;
		case TRANSGFX_TYPE_ANCHOR:    
			active_type = TRANSGFX_TYPE_ANCHOR; 
			gtk_widget_set_sensitive(blend_menu, FALSE);
			gtk_widget_set_sensitive(opacity_slider, FALSE);
			gtk_widget_set_sensitive(opacity_text, FALSE);
			break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			PreviewButtonPress
 *	Prototype:	static void PreviewButtonPress()
 *					
 *	Description:
 *		Determine mouse coordinates when the mouse button is pressed
 *		in the drawing area (preview) widget.  If the mouse press is within
 *		a given distance to the rotate grab box then turn on rotation.
 *		otherwise make sure its turned off.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonPress(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * Turn on rotations.
	 */
	page_rotate = TRUE;

	/*
	 * Update the preview window.
	 */
	PreviewUpdate();
}


/*========================================================================
 *	Name:			PreviewButtonRelease
 *	Prototype:	static void PreviewButtonRelease()
 *					
 *	Description:
 *		Disable rotation when user releases button.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewButtonRelease(
	GtkWidget		*w,
	GdkEventButton	*event
)
{
	page_rotate = FALSE;
}


/*========================================================================
 *	Name:			PreviewUpdate
 *	Prototype:	static void PreviewUpdate()
 *					
 *	Description:
 *		Redraw the preview window.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewUpdate()
{
	int				xoffset, yoffset;
	static GdkGC	*gcWhite = NULL;
	static GdkGC	*gcBlack = NULL;
	static GdkGC	*gcDotted = NULL;
	int				i;
	gfloat			radians;
	int				new_rotation_angle;
	GdkPoint			corners[4];
	char				buf[32];


	if ( preview->widget.window == NULL )
		return;

	gdk_window_clear(preview->widget.window);

	if (gcWhite == NULL)
	{
		gcWhite = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcWhite, GTK_WIDGET(preview)->style->white_gc );
		gcBlack = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcBlack, GTK_WIDGET(preview)->style->black_gc );
		gcDotted = gdk_gc_new(preview->widget.window);
		gdk_gc_copy( gcDotted, GTK_WIDGET(preview)->style->black_gc );
	}

	/*
	 * Draw external circle inside which the page is rotated.
	 */
	xoffset = 2;
	yoffset = 2;
	gdk_draw_arc(
		preview->widget.window, gcBlack, FALSE,
		xoffset,yoffset,
		PREVIEW_WSIZE-4, PREVIEW_HSIZE-4,
		0, 360*64);

	/*
	 * Compute coordinates of the internal page based on the current angle.
	 * The first corner is the lower right of the page and the rest of the
	 * corners are computed in a counter-clockwise pattern.
	 * 
	 * First, convert degrees to radians.
	 * Next, compute current corner.
	 * Finally, compute next angle in radians.
	 */
	for (i=0; i<4; i++)
	{
		new_rotation_angle = rotation_angle+45.0+(i*90.0);
		radians = new_rotation_angle * ( 2.0 * M_PI / 360.0 );

		corners[i].x = cos(radians) * ((PREVIEW_WSIZE - 4) / 2);
		corners[i].y = sin(radians) * ((PREVIEW_HSIZE - 4) / 2);
	}

	/* Conver corners to relative offsets from drawing window origin */
	for (i=0; i<4; i++)
	{
		polycorners[i].x = corners[i].x + ((PREVIEW_WSIZE - 4) / 2) + xoffset;
		polycorners[i].y = corners[i].y + ((PREVIEW_HSIZE - 4) / 2) + yoffset;
	}

	/* Draw the white page */
	gdk_draw_polygon(
		preview->widget.window, gcWhite, TRUE,
		polycorners, 4);
		
	/* Draw corner grab boxes */
	for (i=0; i<4; i++)
	{
		gdk_draw_rectangle(
			preview->widget.window, gcBlack, 0,
			corners[i].x + ((PREVIEW_WSIZE - 4) / 2) - 3, 
			corners[i].y + ((PREVIEW_HSIZE - 4) / 2) - 3, 
			6, 6);
	}

	/* Draw dotted line to original upper right corner of page. */
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_ON_OFF_DASH, 0, 0);
	gdk_draw_line(
		preview->widget.window, gcDotted, 
		((PREVIEW_WSIZE - 4) / 2) + xoffset, 
		((PREVIEW_HSIZE - 4) / 2) + yoffset,
		polycorners[3].x, polycorners[3].y );
	gdk_gc_set_line_attributes(gcDotted, 1, GDK_LINE_SOLID, 0, 0);

	/*
	 * Update angle displayed in text entry field. We need to block the
	 * callbacks for this field first, update the field, then unblock
	 * the callbacks or else we get stuck in an infinite loop.
	 */
	if ( !keypress )
	{
		gtk_signal_handler_block_by_func (GTK_OBJECT (angle_text), 
			(GtkSignalFunc) TextEntryUpdates, (gpointer)NULL);
		sprintf(buf, "%.3f", (gfloat)rotation_angle);
		gtk_entry_set_text(GTK_ENTRY(angle_text), buf);
		gtk_signal_handler_unblock_by_func (GTK_OBJECT (angle_text), 
			(GtkSignalFunc) TextEntryUpdates, (gpointer)NULL);
	}

	/*
	 * Force screen updates.
	 */
	gdk_flush();

}


/*========================================================================
 *	Name:			PreviewMotion
 *	Prototype:	static void PreviewMotion()
 *					
 *	Description:
 *		Update preview window settings based on which item (a margin line
 *		or the grab box) is going to be moved.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
PreviewMotion(
	GtkWidget		*widget,
	GdkEventMotion	*event
)
{
	int		offsetx, offsety;

	/*
	 * Get current mouse location and compute its offset from when the mouse
	 * button was pressed.
	 */
	offsetx = event->x - mouse_x;
	offsety = event->y - mouse_y;

	mouse_x = event->x;
	mouse_y = event->y;

	/*
	 * Check for keyboard modifiers.
	 */
	keymod = 1.0;
	if (event->state & GDK_SHIFT_MASK)
	{
		keymod = 3.0;
	}
	if (event->state & GDK_CONTROL_MASK)
	{
		keymod = 6.0;
	}
	if ( (event->state & GDK_CONTROL_MASK) && (event->state & GDK_SHIFT_MASK) )
	{
		keymod = 10.0;
	}

	/*
	 * If a grab box is enabled, compute new angle.
	 */
	if ( page_rotate )
	{
		rotation_angle += offsetx/keymod + offsety/keymod;
		if ( rotation_angle >  359.0 ) rotation_angle = 0.0;
		if ( rotation_angle < -359.0 ) rotation_angle = 0.0;
	}

	/*
	 * Redraw the screen.
	 */
	PreviewUpdate();

}


/*========================================================================
 *	Name:			ResetFields
 *	Prototype:	static void ResetFields()
 *					
 *	Description:
 *		Reset the dialog to its initial configuration.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ResetFields()
{
	char	buf[16];

	gtk_toggle_button_set_state (
			GTK_TOGGLE_BUTTON (new_layer_radio_button), 
			TRUE);

	gtk_toggle_button_set_state (
			GTK_TOGGLE_BUTTON (smooth_button), FALSE);

	gtk_option_menu_set_history (GTK_OPTION_MENU (blend_menu), 0);

	sprintf(buf, "%.1f", OPACITY_DEFAULT);
	gtk_entry_set_text(GTK_ENTRY(opacity_text), buf);

	active_type = TRANSGFX_TYPE_NEW_LAYER;
	rotation_angle = 0.0;
	blend_mode = 0;

	RebuildLayerMenu();
	PreviewUpdate();
}


/*========================================================================
 *	Name:			RebuildLayerMenu
 *	Prototype:	static void RebuildLayerMenu()
 *					
 *	Description:
 *		Rebuild the layer options menu.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
RebuildLayerMenu()
{
	gtk_widget_hide(layer_menu);
	gtk_widget_destroy(layer_options);
	layer_options = gimp_layer_menu_new(LayerConstraints,
					LayerSelect,
					NULL,
					0);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(layer_menu), layer_options);
	gtk_widget_show(layer_menu);
}


/*========================================================================
 *	Name:			LayerConstraints
 *	Prototype:	static gint LayerConstraints()
 *					
 *	Description:
 *		Callback used to determine which layer we'll include in our
 *		menu of layers.  Basically, we include all of them.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static gint
LayerConstraints(
	gint32	cb_image_id, 
	gint32	drawable_id, 
	gpointer	data
)
{
	if (cb_image_id != image_id)
		return FALSE;
 
	/* No real constraints at this point */
	return TRUE;
}


/*========================================================================
 *	Name:			LayerSelect
 *	Prototype:	static void LayerSelect()
 *					
 *	Description:
 *		What to do when the user selects a particular layer from the menu.
 *		Basically, just save the id of that layer.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
LayerSelect(
	gint32 	id, 
	gpointer	data
)
{
	layer_id = id;
}


/*========================================================================
 *	Name:			BlendSelect
 *	Prototype:	static void BlendSelect()
 *					
 *	Description:
 *	Set the blend mode to the user selected value.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
BlendSelect(
	gint32 	id, 
	gpointer	data
)
{
	blend_mode = (int) data;
}


/*========================================================================
 *	Name:			ArrowCallback
 *	Prototype:	static void ArrowCallback()
 *					
 *	Description:
 *		Handle mouse clicks on arrow buttons.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
ArrowCallback(
	GtkWidget	*widget,
	gpointer		data
)
{
	switch((int)data)
	{
		case TRANSGFX_DOWN: 
			rotation_angle += 0.001;
			if ( rotation_angle > 359.0 ) 
				rotation_angle = 0.0;
			break;

		case TRANSGFX_UP: 
			rotation_angle -= 0.001;
			if ( rotation_angle < -359.0 ) 
				rotation_angle = 0.0;
			break;
	}

	PreviewUpdate();
}


/*========================================================================
 *	Name:			TextEntryUpdates
 *	Prototype:	static void TextEntryUpdates()
 *					
 *	Description:
 *		Update the display based on input from the user.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
TextEntryUpdates(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	angle;

	/*
	 * Block signals and prevent other routines from updating the
	 * input field  while we're doing this update.
	 */
	gtk_signal_handler_block_by_func (GTK_OBJECT (angle_text), 
		(GtkSignalFunc) TextEntryUpdates, (gpointer)NULL);
	keypress = TRUE;

	/*
	 * Get Angle text data.
	 */
	angle = atof( gtk_entry_get_text(GTK_ENTRY(angle_text)) );

	/*
	 * Clamp it between 359 and -359.
	 */
	if ( angle >  359.0 ) angle = 0.0;
	if ( angle < -359.0 ) angle = 0.0;

	rotation_angle = angle;

	PreviewUpdate();

	keypress = FALSE;
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (angle_text), 
		(GtkSignalFunc) TextEntryUpdates, (gpointer)NULL);
}


/*========================================================================
 *	Name:			OpacityTextUpdate
 *	Prototype:	static void OpacityTextUpdate()
 *					
 *	Description:
 *		Handle updates from Opacity text widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
OpacityTextUpdate(
	GtkWidget	*widget,
	gpointer		data
)
{
	gfloat	newvalue;

	/*
	 * Block signals and prevent other routines from updating the
	 * input field  while we're doing this update.
	 */
	gtk_signal_handler_block_by_func (GTK_OBJECT (opacity_text), 
		(GtkSignalFunc) OpacityTextUpdate, (gpointer)NULL);
	keypress = TRUE;

	newvalue = atof(gtk_entry_get_text(GTK_ENTRY(widget)));

	if ((newvalue >= GTK_ADJUSTMENT(opacity_adj)->lower) &&
		(newvalue < GTK_ADJUSTMENT(opacity_adj)->upper))
	{
		GTK_ADJUSTMENT(opacity_adj)->value = newvalue;
		gtk_signal_emit_by_name(opacity_adj, "value_changed");
	}

	keypress = FALSE;
	gtk_signal_handler_unblock_by_func (GTK_OBJECT (opacity_text), 
		(GtkSignalFunc) OpacityTextUpdate, (gpointer)NULL);
}


/*========================================================================
 *	Name:			OpacitySliderUpdate
 *	Prototype:	static void OpacitySliderUpdate()
 *					
 *	Description:
 *		Handle updates from Opacity scale widget.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *	Method:
 *	Restrictions:
 *========================================================================*/
static void
OpacitySliderUpdate(
	GtkAdjustment	*adjustment
)
{
	char		buf[32];

	if ( keypress ) 
		return;

	sprintf(buf, "%.1f", adjustment->value);

	gtk_signal_handler_block_by_data(GTK_OBJECT(opacity_text), NULL);
	gtk_entry_set_text(GTK_ENTRY(opacity_text), buf);
	gtk_signal_handler_unblock_by_data(GTK_OBJECT(opacity_text), NULL);

}


/*========================================================================
 *	Name:			OKCallback
 *	Prototype:	static void OKCallback()
 *					
 *	Description:
 *		Process the image.
 *
 *	Input Arguments:
 *	Output Arguments:
 *	Return Values:
 *
 *	Global Values:
 * active_type			which disposition type to use (new layer or anchor)
 * blend_mode			mode to use for new layers
 * layer_id				which layer to use
 *
 *	Method:
 *	1. Gather field data
 * 2. convert degrees into radians
 * 3. if selection exists, float it.  If not, float entire layer.
 * 4. call gimp_rotate for floating layer.
 * 5. either anchor floating layer or make it a new layer.
 * 6. set layer mode and opacity if new layer requested.
 *
 *	Notes:
 * We don't do much error checking here.  Lets hope things work out for
 * the best!
 *========================================================================*/
static void
OKCallback()
{
	gfloat	angle_degrees, angle_radians;
	int		opacity;
	int		image_id;
	gint32	selection_id;
	gint32	active_layer;
	gint32	active_drawable;
	GimpParam	*return_vals;
	int		nreturn_vals;
	char		disposition[32];
	char		buf[256];
	gint32	*layers;
	gint		nlayers;
	gboolean	found;
	int		i;

	/*
	 * Gather field data not kept in global variables.
	 */
	opacity = atoi( gtk_entry_get_text(GTK_ENTRY(opacity_text)) );
	angle_degrees = atof( gtk_entry_get_text(GTK_ENTRY(angle_text)) );
	image_id = gimp_drawable_image_id(drawable_id);

	/*
	 * Validate the layer id - it may have gone away since the time we
	 * last updated the menu.
	 */
	layers = gimp_image_get_layers(image_id, &nlayers);
	found = FALSE;
	if ( layers != NULL )
	{
		for (i=0; i<nlayers; i++ )
			if ( layers[i] == layer_id )
				found = TRUE;
		g_free(layers);
	}

	if (found == FALSE)
	{
		sprintf(buf, "%s %s",
			"The selected layer seems to have been deleted.  ",
			"Please select another layer.\n");
		GFXMsgWindow(
			buf,
			GFX_ERROR_TYPE | GFX_MSG_NOOK | GFX_MSG_NOHELP,
			NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, 200, 60);

		/*
		 * Rebuild the layer menu.
		 */
		RebuildLayerMenu();

		return;
	}

	/*
	 * Convert degrees into radians.
	 */
	angle_radians = angle_degrees * ( 2 * M_PI / 360 );

	/*
	 * Check for existance of selection.  If it doesn't exist, select
	 * the entire layer.
	 */
	sprintf(buf,"Finding/making selection...\n");
	GFXMsgWindowUpdate( buf, msg_data );
	return_vals = gimp_run_procedure ("gimp_selection_is_empty",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	if (return_vals[0].data.d_status == GIMP_PDB_SUCCESS)
	{
		if (return_vals[1].data.d_int32 != 0)
		{
			gimp_destroy_params (return_vals, nreturn_vals);
			return_vals = gimp_run_procedure ("gimp_selection_all",
									&nreturn_vals,
									GIMP_PDB_IMAGE, image_id,
									GIMP_PDB_END);
		}
	}
	gimp_destroy_params (return_vals, nreturn_vals);
 

	/* 
	 * Float selection.
	 */
	sprintf(buf,"Floating selection...\n");
	GFXMsgWindowUpdate( buf, msg_data );
	return_vals = gimp_run_procedure ("gimp_selection_float",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_DRAWABLE, layer_id,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_INT32, 0,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Retrieve the floating selection's layer id.
	 */
	return_vals = gimp_run_procedure ("gimp_image_floating_selection",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	selection_id = return_vals[1].data.d_layer;
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Get the active drawable for the floating selection.
	 */
	return_vals = gimp_run_procedure ("gimp_image_active_drawable",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_END);
	active_drawable = return_vals[1].data.d_drawable;
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Rotate floating selection.
	 */
	sprintf(buf,"Rotating selection...\n");
	GFXMsgWindowUpdate( buf, msg_data );
	return_vals = gimp_run_procedure ("gimp_rotate",
							&nreturn_vals,
							GIMP_PDB_IMAGE, image_id,
							GIMP_PDB_DRAWABLE, active_drawable,
							GIMP_PDB_INT32, GTK_TOGGLE_BUTTON(smooth_button)->active,
							GIMP_PDB_FLOAT, angle_radians,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Anchor/create new layer from float.
	 */
	if (GTK_TOGGLE_BUTTON(new_layer_radio_button)->active)
	{
		sprintf(buf,"Making selection into a new layer...\n");
		sprintf(disposition, "gimp_floating_sel_to_layer");
	}
	else
	{
		sprintf(buf,"Anchoring selection...\n");
		sprintf(disposition, "gimp_floating_sel_anchor");
	}
	GFXMsgWindowUpdate( buf, msg_data );
	return_vals = gimp_run_procedure (disposition,
							&nreturn_vals,
							GIMP_PDB_LAYER, selection_id,
							GIMP_PDB_END);
	gimp_destroy_params (return_vals, nreturn_vals);

	/*
	 * Get the active layer id.
	 */
	active_layer = gimp_image_get_active_layer(image_id);

	/*
	 * If new layer, set blend mode and opacity.
	 */
	if (GTK_TOGGLE_BUTTON(new_layer_radio_button)->active)
	{
		return_vals = gimp_run_procedure ("gimp_layer_set_mode",
							&nreturn_vals,
							GIMP_PDB_LAYER, active_layer,
							GIMP_PDB_INT32, blend_mode,
							GIMP_PDB_END);
		gimp_destroy_params (return_vals, nreturn_vals);
		gimp_layer_set_opacity(active_layer, opacity);
	}

	/*
	 * Rebuild the layer menu.
	 */
	RebuildLayerMenu();

	/*
	 * Update windows.
	 */
	gimp_displays_flush ();
}

