#!/bin/bash -p
# 
# Copy the appropriate files, in the correct order,
# to the boot partition on the SD card.
# Then, if requested, copy the rootfs to its partition.
#
# Order of install:
# Raspberry Pi firmware:
#     bootcode.bin
#     start.elf
# kernel.img - Linux Kernel
# cmdline.txt - Linux kernel configuration options
# config.txt - Raspberry Pi bootloader configuration options
# ramdisk - Root file system
# ------------------------------------------------

#--------------------------------------------------------------
# Initialization
#--------------------------------------------------------------
BOOTDIR=none
ROOTFSDIR=none
SDROOTFS=/media/bbrootfs
FIRMWARE=firmware

# These file names are configurable
UIMAGE="kernel.img"
RAMDISK="ramdisk.tar"
BOOTSCR="$FIRMWARE/cmdline.txt"
UENV="$FIRMWARE/config.txt"

# These file names are not configurable
BOOTLOADER=$FIRMWARE/bootcode.bin
ELF=$FIRMWARE/start.elf

#--------------------------------------------------------------
# Provide command line usage assistance
#--------------------------------------------------------------
function doHelp
{
    echo ""
    echo "$0 [-b <dir> | -d <device> | -l <file> | -s <file> | -t <file>] | -r <file> "
    echo "where"
    echo "-b <dir>     Directory where boot partition is mounted; Default: $BOOTDIR"
    echo "-d <device>  Device (such as /dev/sdb2) for root file system; Default: $ROOTFSDIR"
    echo "-l <file>    Image file to use for Linux kernel; Default: $UIMAGE"
    echo "-s <file>    Boot script file; Default: $BOOTSCR"
    echo "-t <file>    Text boot script; Default: $UENV"
    echo "-r <file>    Ramdisk image for root file system; Default: $RAMDISK"
    echo ""
    echo "If -b is specified then the boot files will be installed"
    echo "If -d is specified then the ramdisk will be installed"
    echo "Default is to do nothing."
    echo ""
}

#--------------------------------------------------------------
# Read command line arguments
#--------------------------------------------------------------
while getopts ":b:r:u:l:s:t:d:" Option
do
    case $Option in
    b) BOOTDIR=$OPTARG;;
    d) ROOTFSDIR=$OPTARG;;
    l) UIMAGE=$OPTARG;;
    s) BOOTSCR=$OPTARG;;
    t) UENV=$OPTARG;;
    r) RAMDISK=$OPTARG;;
    *) doHelp; exit 0;;
    esac
done


#--------------------------------------------------------------
# Error checking
#--------------------------------------------------------------
if [ "$BOOTDIR" != "none" ]
then
    if [ ! -d $BOOTDIR ]; then
        echo "The boot directory ($BOOTDIR) is not mounted.  Installation aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $UIMAGE ]; then
        echo "Your missing the Linux kernel binary.  Install aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $BOOTLOADER ]; then
        echo "Your missing the $BOOTLOADER binary.  Install aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $ELF ]; then
        echo "Your missing the $ELF binary.  Install aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $UENV ]; then
        echo "Your missing the uEnv.txt script.  Install aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $BOOT ]; then
        echo "Your missing the boot.scr boot script.  Install aborted."
        doHelp
        exit 1
    fi
else
    echo "No boot directory specified. Boot files will not be installed."
fi

if [ "$ROOTFSDIR" != "none" ]
then
    if [ ! -b $ROOTFSDIR ]; then
        echo "The rootfs device ($ROOTFSDIR) is not available.  Installation aborted."
        doHelp
        exit 1
    fi
    if [ ! -f $RAMDISK ]; then
        echo "Your missing the rootfs file ($RAMDISK).  Install aborted."
        doHelp
        exit 1
    fi
else
    echo "No root file system device specified. Ramdisk will not be installed."
fi

if [ "$BOOTDIR" = "none" ] && [ "$ROOTFSDIR" = "none" ]
then
    doHelp
    exit 1
fi

#--------------------------------------------------------------
# Install boot partition files
#--------------------------------------------------------------
if [ "$BOOTDIR" != "none" ]
then
    sudo cp $UIMAGE $BOOTDIR
    sudo cp $BOOTLOADER $BOOTDIR
    sudo cp $ELF $BOOTDIR
    sudo cp $BOOTSCR $BOOTDIR
    sudo cp $UENV $BOOTDIR
fi

#--------------------------------------------------------------
# Install ramdisk image
#--------------------------------------------------------------
if [ "$ROOTFSDIR" != "none" ]
then
    sudo mkdir -p $SDROOTFS
    sudo mount $ROOTFSDIR $SDROOTFS
    sudo tar -C $SDROOTFS -xvf $RAMDISK 
    cwd=`pwd`
    cd $SDROOTFS && sudo chown -R root.root .
    cd $cwd
    sudo umount $SDROOTFS
    sudo rmdir $SDROOTFS
fi

