#!/bin/sh
# Automounter for removable devices under mdev
# Based on block.agent by snafu [Alexander Zangerl (az)]
#     http://www.snafu.priv.at/mystuff/mdev/block.agent
# License: GPL version 1 or version 2
# -------------------------------------------------------------------
# This script mounts at /media/devname.
# It ignores non-removable devices.
# It unmounts if the device is removed.
# Mount points are created and removed as necessary.
# Requirements: busybox and mdev.
# OPTIONFILE (/etc/block.settings) format:
# <UUID> <Label> <fstype> <mount options>
# Examples:
# * * ext3 ro
# * * ext4 ro
# -------------------------------------------------------------------
# Note:
# We're always mounting read-only since we expect there to be
# media for playback on the removable device.
# -------------------------------------------------------------------

# -------------------------------------------------------------------
# LOGGING setup and functions
# -------------------------------------------------------------------
# terminal on stderr -> log to that and to syslog, otherwise syslog only
ME=$(busybox basename $0)"[$$]"
[ -t 2 ] && LOGARGS=-s
mesg () 
{
    busybox logger $LOGARGS -t "$ME" "$@"
}

debug_mesg () 
{
    [ -z "$DEBUG" ] && return
    mesg "$@"
}

# -------------------------------------------------------------------
# Initialization
# -------------------------------------------------------------------
BASE=/media
STATEF=/var/run/block.agent.state
OPTIONFILE="/etc/block.settings"
cd /tmp

# -------------------------------------------------------------------
# Initialization
# -------------------------------------------------------------------
# Finds a mountpoint for specified device (passed as $1)
# Sets $MOUNT to the full mount command, or empty if not to be mounted.
# Sets $TARGET to the mount point.
findmount ()
{
    local RD=$1
    MOUNT=""

    debug_mesg "testing $RD for filesystem"

    # This loads the output from blkid into the environment 
    eval `/sbin/blkid -o udev $RD`

    # Make sure we have a filesystem - ignore if not.
    if [ $? != 0  -o -z "$ID_FS_TYPE" ]; then
        debug_mesg "$RD has no detectable filesystem"
        return
    fi

    # Always use the device name as the mount point.
    TARGET=$BASE/$(busybox basename $RD)
    PREF=$(busybox basename $RD)

    local MOUNTOPT="-"
    # determine where, how and if to mount this thing
    while read U L T O ; do
        [ $(printf %.1s "$U") = "#" ] && continue;
        debug_mesg "testing  $ID_FS_UUID $ID_FS_LABEL $ID_FS_TYPE against $U $L $T"

        if [ \( "$U" = "$ID_FS_UUID" -o "$U" = "*" \) \
            -a \( "$L" = "$ID_FS_LABEL" -o "$L" = "*" \) \
            -a \( "$T" = "$ID_FS_TYPE" -o "$T" = "*" \) ] ; then
            debug_mesg "found entry $U $L $T $O for $RD";
            MOUNTOPT="$O";
            break;
        fi
    done < $OPTIONFILE

    if [ "$MOUNTOPT" = "-" ] ; then
        mesg "$RD: no options available for automounting - skipping."
        return
    else
        mesg "$RD: $TARGET options $MOUNTOPT"
        if [ -n "$MOUNTOPT" ] ; then
            MOUNT="mount -o $MOUNTOPT $RD $TARGET"
        else
            MOUNT="mount $RD $TARGET"
        fi
    fi
    return
}

# -------------------------------------------------------------------
# Choose an action based on what mdev is handling us.
# -------------------------------------------------------------------
# what we get: 
# ACTION=add HOME=/ SEQNUM=2223 PWD=/dev 
# DEVNAME=sdd1 MAJOR=8 MINOR=49 DEVTYPE=partition
# that's from mdev: MDEV=sdd1 SUBSYSTEM=block
# DEVPATH=/devices/pci0000:00/0000:00:1d.7/usb1/1-7/1-7:1.0/host3/target3:0:0/3:0:0:0/block/sdd/sdd1

REALDEV=/dev/$MDEV
if [ "$ACTION" = "add" -o "$ACTION" = "change" ]; then

    # Test if the device is a removable device.
    REMF="/sys/$DEVPATH/removable"
    [ -f "$REMF" ] || REMF="/sys/$DEVPATH/../removable"

    if [ "`cat $REMF`" != 1 ]; then
        debug_mesg "device $REALDEV is not removable."
        exit 0
    fi

    # sets $MOUNT, $TARGET
    findmount $REALDEV
    if [ -z "$MOUNT" ] ; then
        exit 0;
    else
        mesg "trying automount of $REALDEV at $TARGET"
        [ ! -d "$TARGET" ] && mkdir $TARGET
        debug_mesg "$MOUNT"
        eval $MOUNT
        RES=$?
        if [ $RES != 0 ] ; then
            mesg "mounting $REALDEV failed with $RES"
            exit 1
        fi
        echo $REALDEV $TARGET >> $STATEF
    fi
elif [ "$ACTION" = "remove" ] ; then
    MPT=$(egrep "^$REALDEV /" $STATEF|cut -f 2 -d " ")

    # ours? unmount if mounted, remove dir
    if [ -n "$MPT" ]; then

        egrep -v "^$REALDEV $MPT" $STATEF > $STATEF.x
        mv $STATEF.x $STATEF

        LOC=$(egrep "^$REALDEV $MPT" /proc/mounts)
        if [ -n "$LOC" ] ; then
            mesg "attempting late unmount of $REALDEV from $MPT"
            umount -f $MPT
            RES=$?
            if [ $RES != 0 ] ; then
                mesg "unmounting $MPT failed with $RES"
            fi
        fi
        mesg "cleaning up mountpoint dir $MPT for $REALDEV"
        rmdir $MPT
    fi
else
    debug_mesg "ignoring action $ACTION with $MDEV"
    exit 1
fi
exit 0
